/*
Copyright (c) 2003, Michel Jean-Franois <jfmichel(at)operamail(dot)com>
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer. 

- 	Redistributions in binary form must reproduce the above copyright 
	notice, this list of conditions and the following disclaimer in the 
	documentation and/or other materials provided with the distribution. 

-	Neither the name of Sobek nor the names of its contributors 
	may be used to endorse or promote products derived from this software 
	without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/
#ifndef __operandes_hpp
#define __operandes_hpp

#include <ida.hpp>
#include <idp.hpp>

#include "options.hpp"

#ifdef SDK_46
#include <intel.hpp>
#endif

#include <list>

//--------------------------------------------------------------------------
//dclaration
class df_op_t;
//--------------------------------------------------------------------------
// variable externe de debugging
// permet de verifier que tous les objets df_op_t
// crer sont bien effacs  la fin du programme

extern unsigned long leak;        // nombre courrant d'objets df_op_t instancis
extern unsigned long total;       // nombre total d'objets df_op_t crer

#ifdef SDK_46

#define R_none (RegNo)(255)

//---------------------------------
// part of intel.hpp modified in order to test arbitrary ushort

inline bool is_ad16(ushort _auxpref)          // is current addressing 16-bit?
{
  int p = _auxpref & (aux_use32|aux_use64|aux_natad);
  return p == aux_natad || p == aux_use32;
}

inline bool is_ad32(ushort _auxpref)          // is current addressing 32-bit?
{
  int p = _auxpref & (aux_use32|aux_use64|aux_natad);
  return p == (aux_natad|aux_use32)
      || p == 0
      || p == aux_use64;
}

inline bool is_op16(ushort _auxpref)          // is current operand size 16-bit?
{
  int p = _auxpref & (aux_use32|aux_use64|aux_natop);
  return p == aux_natop                                 // 16-bit segment, no prefixes
      || p == aux_use32                                 // 32-bit segment, 66h
    /*|| p == aux_use64 && (cmd.rex & REX_W) == 0*/;      // 64-bit segment, 66h, no rex.w
}

inline bool is_op32(ushort _auxpref)          // is current operand size 32-bit?
{
  int p = _auxpref & (aux_use32|aux_use64|aux_natop);
  return p == 0                                         // 16-bit segment, 66h
      || p == (aux_use32|aux_natop)                     // 32-bit segment, no prefixes
    /*|| p == (aux_use64|aux_natop) && (cmd.rex & REX_W) == 0*/; // 64-bit segment, 66h, no rex.w
}

inline char get_address_dtyp(void)
{
  return ad32() ? dt_dword : dt_word;
}

inline char get_operand_dtyp(void)
{
  return op32() ? dt_dword : op16() ? dt_word : dt_byte;
}
//---------------------------------
/*
inline void set_op16(ushort auxpref)
{
}

inline void set_op32(ushort auxpref)
{
}

inline void set_ad16(ushort auxpref)
{
}
*/
inline void set_ad32(ushort& _auxpref)
{
  _auxpref |= (aux_natad|aux_use32);
}
	

#else

//--------------------------------------------------------------------------
// sdk 4.5 
// Intel 80x86 cmd.auxpref bits
// part of intel.hpp

#define aux_lock        0x0001  // instruction uses prefix lock
#define aux_rep         0x0002  // instruction uses prefix rep
#define aux_repne       0x0004  // instruction uses prefix repne
#define aux_use32       0x0008  // segment type
#define aux_large       0x0010  // offset field is 32-bit (16-bit is not enough)
#define aux_short       0x0020  // short (byte) displacement used
#define aux_prefix      0x0040  // prefix byte is not used
#define aux_op32        0x0080  // op32 prefix byte is not used
#define aux_ad32        0x0100  // ad32 prefix byte is not used
#define aux_basess      0x0200  // SS based instruction
#define aux_op_is_32    0x0400  // operand size 32
#define aux_ad_is_32    0x0800  // addressing mode 32
#define aux_fpemu       0x1000  // FP emulator instruction

// l'instruction utilise des oprandes de 32 bits
inline int op32(void)   { return (cmd.auxpref & aux_op_is_32) != 0; }
// l'instruction utilise des adresses de 32 bits
inline int ad32(void)   { return (cmd.auxpref & aux_ad_is_32) != 0; }

//--------------------------------------------------------------------------
// dfinition des specflags des oprandes
#define hasSIB          specflag1
#define sib             specflag2
//--------------------------------------------------------------------------
// enumration des registres
enum RegNo {
 R_ax = 0,    // 00000
 R_cx = 1,    // 00001
 R_dx = 2,    // 00010
 R_bx = 3,    // 00011
 R_sp = 4,    // 00100
 R_bp = 5,    // 00101
 R_si = 6,    // 00110
 R_di = 7,    // 00111
 R_al = 8,    // 01000
 R_cl = 9,    // 01001
 R_dl = 10,   // 01010
 R_bl = 11,   // 01011
 R_ah = 12,   // 01100
 R_ch = 13,   // 01101
 R_dh = 14,   // 01110
 R_bh = 15,   // 01111

 R_es = 16,   // 10000
 R_cs = 17,   // 10001
 R_ss = 18,   // 10010
 R_ds = 19,   // 10011
 R_fs = 20,   // 10100
 R_gs = 21,   // 10101
 R_none = 255 // 11111111

};
#endif 
//--------------------------------------------------------------------------
// valeur du sib byte pour une oprande de la forme [ESP+XXX]
#define sib_esp         36
//--------------------------------------------------------------------------
// liste_ptrop : liste de pointeurs d'oprandes
//--------------------------------------------------------------------------

class liste_ptrop : public list<df_op_t*>
{
public:
  int delta_total; // somme des deltas ajouts dans basic_ana::adjust_operands()
  liste_ptrop(void) { delta_total = 0; }
  bool vide();
  bool find_op(df_op_t*) const;
};

//--------------------------------------------------------------------------
// liste_op : liste d'oprandes
//--------------------------------------------------------------------------

typedef list<df_op_t> liste_op;

//--------------------------------------------------------------------------
// df_op_t : Data Flow op_t
//--------------------------------------------------------------------------

// hrite de op_t

class df_op_t : public op_t
{

  bool get_operand_registers(RegNo *base, RegNo *index, char *dtyp) const;

public:
  ulong cs;             // permet d'valuer si le segment de code est le mme
  ushort auxpref;       // ncessaire pour tester le type d'adressage
  liste_ptrop* aliases; // pointeur sur la liste des alias de l'oprande
                        // les objets points ne sont pas dtruits par
                        // le destructeur
  df_op_t* aliased;     // pointe sur l'oprande dont this est un alias
  ea_t i_ea;            // adresse de l'instruction qui utilise l'oprande
  bool seek_in_phrase;  // seulement si l'oprande est un registre.
                        // dans le cadre de l'oprateur <=, ce flag indique
                        // si ce registre est compar aux registres qui forment
                        // les oprandes de type o_mem, o_phrase et o_displ
                        // (utilis dans trace_backward)
  bool norm;            // indique que l'oprande est normalise
  bool hide;            // indique si l'operande est cache
  int hide_depth;       // indique  quelle profondeur l'oprande a t cache

//--------------------------------------------------------------------------

  // constructeurs

  df_op_t();
  // construit une oprande
  //  partir du numro de l'oprande de l'instruction courante
  df_op_t(int opnum);
  df_op_t(ea_t ea, RegNo r, char data_type);  // construit un registre
  df_op_t(const df_op_t &src); // copy constructeur

  // destructeur
  ~df_op_t();

//--------------------------------------------------------------------------
  // operateurs de comparaison

  bool operator<=(const df_op_t&) const; // this est incluse ou gale  op
  bool operator>=(const df_op_t&) const; // op est incluse ou gale  this
  bool operator==(const df_op_t&) const; // this est gale  op
  bool operator!=(const df_op_t&) const; // this est diffrent de op

//--------------------------------------------------------------------------
  // fonctions membres

  short getReg()const; // renvoie le numro d'un registre
  void normalize(); // normalise l'oprande

  // ajoute les registres utiliss par le mode d'adressage de l'oprande
  // dans la liste
  int createRegsFromPhrase(liste_ptrop&) const;

  // ajoute les numros de registres utiliss par le mode d'adressage
  // de l'oprande dans la liste
  int getRegsFromPhrase(list<RegNo>&) const;


  void affiche(); // affiche une oprande dans la fentre de messages d'Ida

  // renvoie le pointeur d'une chane de caractre reprsentant l'oprande
  const char* getTxt(long delta=0);

  void setAliased(df_op_t*); // Dfinit l'oprande dont this est l'alias

  // ajoute le pointeur pass en paramtre  la liste des alias de l'oprande
  void addAlias(df_op_t*);

  // retire le pointeur pass en paramtre de la liste des alias de l'oprande
  void remAlias(df_op_t*);

  // crer l'oprande de pile implicite des instructions push et pop
  df_op_t* create_normalized_operand(ulong offset);
};

#endif
