/*
Copyright (c) 2003, Michel Jean-Franois <jfmichel(at)operamail(dot)com>
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer. 

- 	Redistributions in binary form must reproduce the above copyright 
	notice, this list of conditions and the following disclaimer in the 
	documentation and/or other materials provided with the distribution. 

-	Neither the name of Sobek nor the names of its contributors 
	may be used to endorse or promote products derived from this software 
	without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#include "operandes.hpp"
#include <ua.hpp>
#include <frame.hpp>

//--------------------------------------------------------------------------
//  aide  l'analyse des champs R/M Field et SIB
//--------------------------------------------------------------------------

struct add_reg_t
{
  RegNo reg1;
  RegNo reg2;
};

// adressage 16 bits

// o_phrase
static const add_reg_t rmField_16p[] = {
// Index     Base      r/m field
   {R_si   , R_bx}   , // 000
   {R_di   , R_bx}   , // 001
   {R_si   , R_bp}   , // 010
   {R_di   , R_bp}   , // 011
   {R_si   , R_none} , // 100
   {R_di   , R_none} , // 101
   {R_none , R_none} , // 110
   {R_none , R_bx}     // 111
};

// o_displ
static const add_reg_t rmField_16d[] = {
// Index     Base      r/m field
   {R_si   , R_bx}   , // 000
   {R_di   , R_bx}   , // 001
   {R_si   , R_bp}   , // 010
   {R_di   , R_bp}   , // 011
   {R_si   , R_none} , // 100
   {R_di   , R_none} , // 101
   {R_none , R_bp}   , // 110
   {R_none , R_bx}     // 111
};


// adressage 32 bits sans sib

static const add_reg_t rmField_32[] = {
// o_phrase  o_displ    r/m field
   {R_ax   , R_ax }   , // 000
   {R_cx   , R_cx }   , // 001
   {R_dx   , R_dx }   , // 010
   {R_bx   , R_bx }   , // 011
   {R_none , R_none } , // 100
   {R_none , R_bp }   , // 101
   {R_si   , R_si }   , // 110
   {R_di   , R_di }     // 111
};

// adressage 32bits avec sib

static const add_reg_t sib_table[] = {
// Index     Base      Sib byte: index/base fields
   {R_ax   , R_ax}   , // 000
   {R_cx   , R_cx}   , // 001
   {R_dx   , R_dx}   , // 010
   {R_bx   , R_bx}   , // 011
   {R_none , R_sp}   , // 100
   {R_bp   , R_none} , // 101
   {R_si   , R_si}   , // 110
   {R_di   , R_di}     // 111
};


//--------------------------------------------------------------------------
// liste_ptrop : liste d'oprandes
//--------------------------------------------------------------------------

// renvoie true si la liste est vide
//                 ou ne contient que des oprandes caches (hide == true)
//         false sinon
bool liste_ptrop::vide()
{
  bool ret = true;

  if (!empty())
  {
    bool only_hide = true ;
    for(liste_ptrop::iterator p = begin(); p != end() && only_hide; ++p)
    {
      if (!(*p)->hide)
        only_hide = false;
    }
    if (!only_hide)
      ret = false;
  }
  return ret;
}

//--------------------------------------------------------------------------
// renvoie true si l'oprande pointe par op se trouve dans this

bool liste_ptrop::find_op(df_op_t* op) const
{
  bool found = false;

  for(liste_ptrop::const_iterator p = this->begin();
      (p != this->end()) && (!found);
      ++p)
  {
    if (*op == **p)
      found = true ;
  }
  return(found);
}

//--------------------------------------------------------------------------
// df_op_t : data flow op_t
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Constructeurs
//--------------------------------------------------------------------------

df_op_t::df_op_t()
{
  leak++;   // met  jour le nombre courant et total d'oprandes cres
  total++;

  memset(this, 0, sizeof(*this)); // initialise tous les champs de l'objet  0
}

//--------------------------------------------------------------------------
// construit une oprande
//  partir du numro de l'oprande de l'instruction courante

df_op_t::df_op_t(int opnum) :
   op_t(cmd.Operands[opnum]), auxpref(cmd.auxpref), cs(cmd.cs),
   i_ea(cmd.ea), aliases(NULL), aliased(NULL), seek_in_phrase(true),hide(false)
   ,norm(false)
{
  leak++;  // met  jour le nombre courant et total d'oprandes cres
  total++;
}

//--------------------------------------------------------------------------
// construit un registre  partir :
// d'une adresse (ea), un numro de registre (r) et une taille (data_type)
// attention par defaut seek_in_phrase est false !

df_op_t::df_op_t(ea_t ea, RegNo r, char data_type)
{
  leak++;  // met  jour le nombre courant et total d'oprandes cres
  total++;

  memset(this, 0, sizeof(*this));
  type = o_reg;
  dtyp = data_type;
  reg  = r;
  i_ea = ea;
  cs   = BADADDR;
  if (data_type == dt_dword)
    #ifdef SDK_46
    auxpref = aux_use32 | aux_natop;
    #else
    auxpref = aux_op_is_32;
    #endif
  else
    auxpref = 0;
}

//--------------------------------------------------------------------------
// constructeur de copie
// copie une oprande sans copier ses alias

df_op_t::df_op_t(const df_op_t &src) :
  op_t(src), cs(src.cs), auxpref(src.auxpref), i_ea(src.i_ea),
  seek_in_phrase(src.seek_in_phrase), hide_depth(src.hide_depth),hide(src.hide)
  ,norm(src.norm)
{
  leak++;  // met  jour le nombre courant et total d'oprandes cres
  total++;

  aliases = NULL;  // ne copie pas les alias
  aliased = NULL;  //
}

//--------------------------------------------------------------------------
// Destructeur

df_op_t::~df_op_t()
{
  leak--;  // dcrmente le nombre d'oprandes courantes

  delete aliases; // efface la liste d'alias
  aliases = NULL; // (sans effacer les objets points par la liste)
  aliased = NULL; // nous ne possedons pas cette oprande
}

//--------------------------------------------------------------------------
// oprateurs de comparaisons
//--------------------------------------------------------------------------
// <=
// teste si this est inclus ou gal  rhs
// renvoie true si l'oprande this est gale a rhs
//              ou si this est un registre et est utilis
//                 par le mode d'adressage de rhs
//         false sinon
//
// ce tableau montre les combinaisons comparables
//
//  This              :    rhs
// ===========================================================================
//  o_reg             :    o_reg                 // galit stricte
//                         o_mem (seulement sib) // ou this est utilis
//                         o_phrase              // par le mode d'adressage
//                         o_displ               // de rhs
//
//  o_imm             :    o_imm                 // galit stricte
//
//  o_near            :    o_near                // galit stricte
//                         o_far
//
//  o_far             :    o_near                // galit stricte
//                         o_far
//
//  o_mem             :    o_mem                 // galit stricte
//
//  o_phrase          :    o_phrase              // galit au deplacement prs
//                         o_displ
//
//  o_displ           :    o_displ               // galit stricte

bool df_op_t::operator<=(const df_op_t& rhs) const
{
  bool ret = false;
  switch (this->type)
  {
    case o_reg:
      switch (rhs.type)
      {
      	case o_reg:   // o_reg <= o_reg
          {
            ret = this->getReg() == rhs.getReg();
          }
      	  break;

        case o_mem:
        // o_reg <= o_mem, cas particulier o un o_mem est en fait un o_displ
          if (!rhs.sib)
            break;
          // no break

      	case o_phrase:  // o_reg <= o_phrase
        case o_displ:   // o_reg <= o_displ
          if (this->seek_in_phrase && !rhs.norm)
          {
            RegNo base;
            RegNo index;
            char dtyp;
            rhs.get_operand_registers(&base, &index, &dtyp);
            ret = (this->reg == base) || (this->reg == index);
          }
      	  break;
      }
      break;

    case o_mem:
      switch (rhs.type)
      {
      	case o_mem: // o_mem <= o_mem
      	  ret = (this->cs == rhs.cs) && (this->addr == rhs.addr);
      	  break;
      }
      break;

    case o_displ: // o_displ <= o_displ
      if ( rhs.type != o_displ || this->addr != rhs.addr )
        break;
      // no break

    case o_phrase: // o_phrase <= o_phrase, o_phrase <= o_displ
      if ( (rhs.type == o_phrase) || (rhs.type == o_displ) )
      {
      	ret = ( phrase == rhs.phrase )
      	   #ifdef SDK_46
      	   && (is_ad32(auxpref) == is_ad32(rhs.auxpref))
      	   #else
           && ((auxpref & aux_ad_is_32) == (rhs.auxpref & aux_ad_is_32) )
           #endif
           && ( hasSIB == rhs.hasSIB)
           && ((hasSIB && (sib == rhs.sib)) || !hasSIB )
           && ( norm == rhs.norm );
      }
      break;

    case o_imm: // o_imm <= o_imm
      ret = rhs.type == o_imm
         && this->value == rhs.value
         && this->dtyp  == rhs.dtyp;
      break;

    case o_far: // o_far <= o_near, o_far_ <= o_far
    case o_near: // o_near <= o_near, o_near <= o_far
      if ( (rhs.type == o_far) || (rhs.type == o_near) )
      	ret = (this->cs == rhs.cs)
      	   && (this->addr == rhs.addr)
      	   && (this->specval == rhs.specval);
      break;

    default:
      break;
  }
  return ret;
}

//--------------------------------------------------------------------------

// teste si *this est compris ou est egal  rhs

bool df_op_t::operator>=(const df_op_t& rhs) const
{
  return rhs <= *this;
}

//--------------------------------------------------------------------------
// test l'galit des champs correspondant deux  deux
// les champs comments ne sont pas pris en compte

bool df_op_t::operator==(const df_op_t& rhs) const
{
  if ( norm != rhs.norm )
  {
    return false;
  }
  return (   (*this).type           == rhs.type
         &&  (*this).dtyp           == rhs.dtyp
         &&  (*this).reg            == rhs.reg
         &&  (*this).value          == rhs.value
         &&  (*this).addr           == rhs.addr
         &&  (*this).specval        == rhs.specval
         &&  (*this).specflag1      == rhs.specflag1
         &&  (*this).specflag2      == rhs.specflag2
         &&  (*this).specflag3      == rhs.specflag3
         &&  (*this).specflag4      == rhs.specflag4
         &&  (*this).auxpref        == rhs.auxpref
         &&  (*this).cs             == rhs.cs
         &&  (*this).hide           == rhs.hide

      // Champs non utiliss

      // &&  (*this).n              == rhs.n
      // &&  (*this).i_ea           == rhs.i_ea
      // &&  (*this).flags          == rhs.flags
        );
}

//--------------------------------------------------------------------------
// vrifie l'ingalit de deux oprandes

bool df_op_t::operator!=(const df_op_t& rhs) const
{
  return !(*this == rhs);
}

//--------------------------------------------------------------------------
// Mthodes et fonctions
//--------------------------------------------------------------------------

// normalise les oprandes du type [ebp+x] en [esp+y]
// op : oprande  normaliser
// f : pointeur de fonction dans laquelle l'oprande est utilise

inline void replace_bp_by_sp(df_op_t *op, func_t *f)
{
  if (f->flags & FUNC_FRAME) // bp based frame
  {
    op->reg    = R_sp;
    op->addr  += f->frsize; // ajuste le dplacement
    op->hasSIB = true;
    op->sib    = 36;        // 00100100b = [esp+displ]
    op->norm   = true;      // normalis = true
    #ifdef SDK_46
    set_ad32(op->auxpref);
    #else
    op->auxpref |= aux_ad_is_32;
    #endif  
  }
}
//--------------------------------------------------------------------------
void df_op_t::normalize()
{
// normalise les oprandes de type [ebp+x] et [esp+x] (o_displ)
//                         en [esp+y]
//
//
  func_t* f = get_func(i_ea);

  // si l'oprande est de type o_displ ou o_phrase, f est != de NULL
  // et si l'op n'est pas dja normalise: on continue
  if ((type == o_displ || type == o_phrase) && f != NULL && !norm)
  {
    #ifdef SDK_46
    if ( !is_ad32(auxpref) )
    #else
    if ( !(auxpref & aux_ad_is_32) )
    #endif
    {
      // -----------------
      // adressage 16 bits
      // -----------------
      if ( (rmField_16d[phrase & 7].reg1 == R_none)
        && (rmField_16d[phrase & 7].reg2 == R_bp) )
      {
        // oprande de type [EBP+xxx]
        replace_bp_by_sp(this, f);
      }
    }
    else if ( !(hasSIB) )
    {
      // --------------------------
      // adressage 32 bits sans SIB
      // --------------------------
      if (rmField_32[phrase & 7].reg2 == R_bp)
      {
      	// [EBP+xxx]
        replace_bp_by_sp(this, f);
      }
    }
    else
    {
    // --------------------------
    // adressage 32 bits avec SIB
    // --------------------------

      if ( (sib_table[ sib       & 7].reg2/*base*/  == R_sp)
        && (sib_table[(sib >> 3) & 7].reg1/*index*/ == R_none) )
      {
        // [ESP+xxx]
        this->addr += get_spd(f,i_ea) + f->frregs + f->frsize; // ajuste l'addr
        norm = true;
      }
      else
      if ( (sib_table[ sib       & 7].reg2/*base*/  == R_none)
        && (sib_table[(sib >> 3) & 7].reg1/*index*/ == R_bp) )
      {
        // [EBP+xxx]
        replace_bp_by_sp(this, f);
      }
    }
  }
}
//--------------------------------------------------------------------------
// modifie les paramtres base, index et dtyp par les valeurs correspondantes
// utilises par le mode d'adressage de l'oprande.
// base  : registre de base de l'adressage
// index : registre d'index de l'adressage
// dtyp  : taille de l'adressage
// renvoie true si l'oprande utilise un mode d'adressage bas et/ou index
//         false sinon
bool df_op_t::get_operand_registers(RegNo *base, RegNo *index, char *dtyp) const
{
  *base  = R_none;
  *index = R_none;
  *dtyp  = dt_dword;
  /*ce 3eme cas est defini comme etant un o_mem meme si on aurait
    pu penser que c'est en fait un o_displ ex: word_1008B00[ebx*2]*/
  if ( type == o_displ || type == o_phrase || ( type == o_mem && sib ))
  {
    #ifdef SDK_46
    if ( !is_ad32(auxpref) )
    #else
    if ( !(auxpref & aux_ad_is_32) )
    #endif
    {
      // -----------------
      // adressage 16 bits
      // -----------------
      const add_reg_t *table = type == o_phrase
                             ? rmField_16p : rmField_16d;

      	*base  = table[phrase & 7].reg1;
        *index = table[phrase & 7].reg2;

      *dtyp  = dt_word;
    }
    else if ( !(hasSIB) )
    {
      // --------------------------
      // adressage 32 bits sans SIB
      // --------------------------
      if (type == o_displ)
        *base = rmField_32[phrase & 7].reg2;
      else
        *base = rmField_32[phrase & 7].reg1;
      *index = R_none;
    }
    else
    {
      // --------------------------
      // adressage 32 bits avec SIB
      // --------------------------
      *base  = sib_table[sib & 7].reg2;
      *index = sib_table[(sib >> 3) & 7].reg1;
    }
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------
// ajoute les numros de registres utiliss par l'oprande
// dans la liste passe en paramtre

int df_op_t::getRegsFromPhrase(list<RegNo> &l) const
{
  RegNo base, index;
  char dtyp;

  l.clear();
  if ( get_operand_registers(&base, &index, &dtyp) )
  {
    if ( base != R_none )
      l.push_front(base);
    if ( index != R_none )
      l.push_front(index);
  }
  return l.size();
}

//--------------------------------------------------------------------------
// cre et ajoute les registres utiliss par le mode d'adressage de l'oprande
// dans la liste l de pointeurs d'oprandes passe en paramtre
// renvoie le nombre de registres ajouts  la liste

int df_op_t::createRegsFromPhrase(liste_ptrop &l) const
{
  RegNo base, index;
  char dtyp;

  int count = 0;
  if ( get_operand_registers(&base, &index, &dtyp) )
  {
    if (base != R_none)
    {
      l.push_front(new df_op_t(i_ea, base, dtyp));
      count++;
    }

    if (index != R_none)
    {
      l.push_front(new df_op_t(i_ea, index, dtyp));
      count++;
    }
  }
  return count;
}

//--------------------------------------------------------------------------
// Affiche les valeurs des champs de l'oprande
// dans la fentre de messages d'Ida

void df_op_t::affiche(void)
{
  msg("_________________________\n");
  msg("n           : %d\n",n);
  msg("va          : 0x%x\n",i_ea);
  msg("type        : %d\n",type);
  msg("flags       : %d\n",flags);
  msg("phrase/reg  : %d\n",phrase);
  msg("value       : %d\n",value);
  msg("dtyp        : %d\n",dtyp);
  msg("specval     : %d\n",specval);
  msg("auxpref     : %d\n",auxpref);
  msg("hasSIB      : %d\n",hasSIB);
  msg("SIB         : %d\n",sib);
  msg("normalized  : %d\n",norm);
  msg("addr        : %d\n",addr);
  msg("aliases     : %x\n",aliases);
  msg("aliased     : %x\n",aliased);
  msg("_________________________\n");
}

//--------------------------------------------------------------------------
// renvoie le pointeur d'une chane de caractre reprsentant l'oprande.
// si l'oprande est affiche par ida,
// la fonction extrait sa reprsentation du listing.
// sinon il faut crer nous mme sa reprsentation.
// seuls les types o_reg o_phrase et o_displ sont grs.
// le paramtre delta est retranch des oprandes normalises
// non affiches par ida pour obtenir le dplacement d'origine.
// les oprandes caches sont suivies du symbol '*'
const char* df_op_t::getTxt(long delta)
{
  static string txt;
  char buffer[MAXSTR];

  if (this->showed()) // si l'oprande est affiche par ida
  {
    ua_ana0(i_ea);
    #ifdef SDK_46
    ua_outop(i_ea,buffer,MAXSTR,n); // remplit buffer avec la reprsentation de l'op n
    #else
    ua_outop(i_ea,buffer,n); // remplit buffer avec la reprsentation de l'op n    
    #endif
    tag_remove(buffer,buffer,sizeof(buffer)); // limine les tags
    txt = buffer;
  }
  else                // sinon il faut la crer
  {
    txt = "NotImplemented";
    switch (type)
    {
      case o_reg:
        if (dtyp == dt_dword)
          txt = "e"; // registre tendu de 32 bits
        txt += ph.regNames[reg];
        break;
      case o_phrase:
      case o_displ:
        {
          list<RegNo> regs;
          getRegsFromPhrase(regs);
          txt = "[";
          // parcours des registres formants l'adressage
          for ( list<RegNo>::iterator p=regs.begin(); p != regs.end(); ++p )
          {
            if ( p != regs.begin() )
              txt += "+";
            if (dtyp == dt_dword)
              txt += "e";
            txt += ph.regNames[*p];
          }
          if (type == o_displ)
          {
            long v = addr;

            if (norm)
              v -= delta;

            if ( v < 0 )
            {
              txt += "-";
              v = -v;
            }
            else
            {
              txt += "+";
            }
            qsnprintf(buffer, sizeof(buffer), "0x%X", v);
            txt += buffer;
          }
          txt += "]";
        }
        break;
      case o_void:
        txt.clear();
        break;
    }
  }

  if(hide)
    txt += "*";
  return txt.c_str();
}

//--------------------------------------------------------------------------
// renvoie le numro de registre des oprandes de type o_reg
//         -1 si l'oprande n'est pas un registre
// les registres de 8 bits sont convertis en leur quivalent 16 bits

short df_op_t::getReg() const
{
  ushort tmp = -1;
  if (type == o_reg)
  {
    tmp = reg;
    // tranforme les registres 8 bits en leur quivalent 16 bits
    if ( (reg <= R_bh) && (reg >= R_al) )
      tmp &= 3;
  }
  return tmp;
}
//--------------------------------------------------------------------------
// Dfinit l'oprande dont this est l'alias

void df_op_t::setAliased(df_op_t *op)
{
  aliased = op;
}
//--------------------------------------------------------------------------
// ajoute le pointeur pass en paramtre  la liste des alias de l'oprande
// si la liste n'existe pas, la mthode la cre
void df_op_t::addAlias(df_op_t *op)
{
  if (aliases == NULL)
    aliases = new liste_ptrop();
  aliases->push_front(op);
}

//--------------------------------------------------------------------------
// retire le pointeur pass en paramtre de la liste des alias de l'oprande
// si la liste est vide, la mthode dtruit la liste
void df_op_t::remAlias(df_op_t *op)
{
  if (aliases != NULL)
  {
    aliases->remove(op);
    if (!aliases->size())
    {
      delete aliases;
      aliases = NULL;
    }
  }
}
//--------------------------------------------------------------------------
// crer l'oprande de pile implicite des instructions push et pop
// l'oprande cre est de type [esp+x]
// offset est le dplacement x
// this est l'oprande source ou destination des instructions push et pop
// l'oprande est ensuite normalise
// renvoie un pointeur vers l'oprande cre

df_op_t* df_op_t::create_normalized_operand(ulong offset)
{
  df_op_t *op = new df_op_t;
  memset(op, 0, sizeof(*op));

  op->cs      = this->cs;
  op->i_ea    = this->i_ea;
  #ifdef SDK_46
  op->auxpref = this->auxpref ;
  set_ad32(op->auxpref);
  #else
  op->auxpref = this->auxpref | aux_ad_is_32;
  #endif
  op->n       = this->n+1;
  op->flags   = this->flags & ~OF_SHOW;       // operande cache
  op->dtyp    = this->dtyp;

  op->type    = o_displ;
  op->reg     = R_sp;
  op->addr    = offset;
  op->hasSIB  = true;
  op->sib     = sib_esp;

  op->seek_in_phrase = true;

  op->normalize();
  return op;
}
