/*
Copyright (c) 2003, Michel Jean-Franois <jfmichel(at)operamail(dot)com>
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer. 

- 	Redistributions in binary form must reproduce the above copyright 
	notice, this list of conditions and the following disclaimer in the 
	documentation and/or other materials provided with the distribution. 

-	Neither the name of Sobek nor the names of its contributors 
	may be used to endorse or promote products derived from this software 
	without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#include "global_ana.hpp"

namespace global_ana
{
  //--------------------------------------------------------------------------
  // variables globales
  static list<area_t> limits;       // intervalle d'adresses limitant l'analyse
  static pair<int,int> depth_limits; // profondeur limitant l'analyse

  //--------------------------------------------------------------------------
  // tablit les limites de profondeur
  // min : la plus petite limite
  // max : la plus grande limite

  void set_depth_limit(const int min,const int max)
  {
    if ((min <= max))
    {
      depth_limits.first = min;
      depth_limits.second = max;
    }
  }
  //--------------------------------------------------------------------------
  // teste si 'depth' est compris dans l'intervalle
  // de profondeur permis.

  bool check_depth_limit(const int _depth)
  {
    return ((_depth >= depth_limits.first) && (_depth <= depth_limits.second));
  }

  //--------------------------------------------------------------------------
  // Teste si les deux listes de pointeurs d'oprandes passes en paramtres
  // pointent sur les mmes oprandes
  // cette fonction ne teste pas les alias des oprandes
  bool isEqual(const liste_ptrop& list1, const liste_ptrop& list2)
  {
    bool equal = false;
    if (list1.size() == list2.size())  // teste la taille
    {
      liste_ptrop::const_iterator p;
      // essaye de trouver chaque lment de list1 dans list2
      for(p = list1.begin(); (p != list1.end()) && (list2.find_op(*p)); ++p)
        ;

      if (p == list1.end()) // si tous les lments sont trouvs
        equal = true;
    }
    return equal;
  }
  //--------------------------------------------------------------------------
  // teste si l'adresse passe en paramtre est dans l'intervalle permis
  bool check_limit(const ea_t ea)
  {
    bool found = false;
    for(list<area_t>::iterator p = global_ana::limits.begin();
        p != global_ana::limits.end() && !found;
        ++p)
    {
      if (ea >= (*p).startEA && ea < (*p).endEA)
        found = true;
    }
    return found;
  }

  //--------------------------------------------------------------------------
  // ajoute un intervalle d'adresses  la liste des intervalles permis
  void add_limits(const area_t area)
  {
    limits.push_front(area);
  }

  //--------------------------------------------------------------------------
  // vrifie l'galit de deux netnodes
  // le test se base sur la structure utilise pour reprsenter les rsultats
  // altval(-1) = le nombre de lignes du netnode
  // supval(['0'..'4']) reprsente les infos sous forme de chane de car.
  // '0' = DF_ALIAS
  // '1' = CF_CHG
  // '2' = DF_COPY
  // '3' = DF_DESTR
  // '4' = CF_USE

  bool isEqual(netnode a, netnode b)
  {
    bool ret = false;
    bool equal = true;
    if ( a.altval(-1) == b.altval(-1) ) // le mme nombre de lignes
    {
      int line;
      for(line = 0; (line < a.altval(-1)) && equal; ++line)
      {
        if(a.altval(line) == b.altval(line)) // teste chaque couple d'adresses
        {
          string s1("");
          string s2("");
          for (int event = '0'; (event < '5') && (equal); ++event)
          { // pour chaque vnement possible
            if(a.supval(event))
              s1 = string(a.supval(line,event));

            if (b.supval(event))
              s2 = string(b.supval(line,event));
            if (s1 != s2)  // teste l'galit des chanes
            {
              equal = false;
            }
          }
        }
        else
            equal = false;
      }
      if ((line == a.altval(-1)) && (equal))
         ret = true;
    }
    return ret;
  }
  //--------------------------------------------------------------------------
  // ajoute un netnode res  la liste de netnode gti
  // l'ajout ne se fait que si aucun des netnodes de gti n'est gal  res
  // selon les critres de la fonction isEqual
  // sinon res est dtruit
  bool add_if_unique(/*IN/OUT*/netnode res, /*IN/OUT*/liste_node* gti)
  {
    bool ret = false;

    if (res.altval(-1)) // si res contient au moins une info
    {
      liste_node::iterator p = gti->begin();
      bool AlreadyThere = false;
      for(; (p != gti->end()) && !AlreadyThere; ++p)
      {
        AlreadyThere = isEqual(res,*p); // teste si res n'est pas dj dans gti
      }
      if (!AlreadyThere)
      {
        gti->push_back(res);
        ret = true;
      }
      else
        res.kill();
    }
    else
      res.kill();    // sinon delete
    return ret;
  }


  //--------------------------------------------------------------------------
  // teste si le bloc de base
  // caractris par les paramtres 'ea' (adresse de dbut) et 'l' (liste_ptrop)
  // se trouve dans la liste de blocs de base 'bb'

  bool find_bb( /*IN*/  const liste_bb* bb,
                /*IN*/  const ea_t ea,
                /*IN*/  const liste_ptrop* l)
  {

    bool found = false;

    for (liste_bb::const_iterator p = bb->begin();
         (p != bb->end()) && !found;
         ++p)
    { // pour chaque lment de la liste de basic_block
   	  if ( (ea == (*p).ea) && isEqual(*l,*(*p).in) )
   	  { // positionne found  true
        found = true;
      }
    }
    return found;
  }
  //--------------------------------------------------------------------------
  // ajoute  working set les blocs de base suivants non encore analyss
  // de la fin d'un bloc de base dtermin par  :
  //
  // trace : liste des oprandes vivantes   la fin d'un bloc de base
  // next : liste des adresses suivantes d'un bloc de base
  // flag : tat par lequel le bloc de base passe le flux de contrle
  // ret_addr : pile des adresses de retour
  // prev : adresse du bloc de base prcdant
  // depth : profondeur des blocs de base suivants
  //
  // Les autres paramtres sont :
  //
  // working_set : liste des blocs de base  analyser
  // history : historique des blocs de base dj analyss
  // mode : TRACE_FORWRAD ou TRACE_BACKWARD


  void feed_working_set(/*IN*/      liste_ptrop* trace,
                        /*IN*/      list<ea_t> &next,
                        /*IN/OUT*/  liste_bb history,
                        /*IN/OUT*/  liste_bb &working_set,
                        /*IN*/      char flag,
                        /*IN*/      liste_ret &ret_addr,
                        /*IN*/      const int mode,
                        /*IN*/      ea_t prev,
                        /*IN*/      int &depth )
  {
    if ( !(trace->empty()))
    {
      bool use_trace = false;

      // use_trace indique si on doit garder la liste trace
      // si true on conserve trace, si false on l'efface

      // pour chaque bloc de base suivant
      for (list<ea_t>::iterator p = next.begin(); p != next.end(); ++p)
      {
        bool ignore_call = false ;

        // si la combinaison next/trace n'a pas encore t analyse
        if (!find_bb(&history, *p, trace))
        {
          // si le bloc se trouve  des limites d'adresses et de profondeurs
          // permises
          if (check_limit(*p) && check_depth_limit(depth))
          {
            use_trace = true;
            working_set.push_front(basic_block(*p,
                                               trace,
                                               flag,
                                               prev,
                                               ret_addr,
                                               depth));
          }
          else // adresse ou profondeur interdite
          {    // si c un NEXT_CALL,
               // on doit le faire passer pour un NEXT_FALSE_CALL
            if (flag == NEXT_CALL)
            {
              ignore_call = true;
            }
            // else sinon rien a faire
          }
        }
        // la combinaison next/trace a dj t analyse
        else // Si le bloc de base passe la main par un NEXT_CALL,
             // il faut en faire un FALSE_CALL vers la suite
             // qu'on a peut tre pas encore analyse
        {    // Dans le cas d'un NEXT_JMP ou NEXT_RET
             // on est certain de ne pas devoir analyser la suite
          if (flag == NEXT_CALL)
          {
            ignore_call = true;
          }
          // else sinon ne rien faire
        }

        if (ignore_call)
        {
          if (!ret_addr.empty()) // pile des adresses de retour vide ?
          {
            flag = NEXT_FALSE_CALL;
            ea_t suiv = ret_addr.front().ea;

            if (mode == TRACE_BACKWARD) // si on trace en backward
            {
              func_t* f = get_func(suiv);

              // si le call est la premire instruction
              if (suiv == f->startEA)
              {
                flag = NEXT_RET; // force un return
              }
              else // ajuste l'adresse de retour en backward
                suiv = decode_prev_insn(suiv);
            }

            if (suiv != BADADDR)
            {
              if (   check_limit(suiv)
                  && check_depth_limit(ret_addr.front().depth))
              {
                ret retour = ret_addr.front();
                ret_addr.pop_front();
                use_trace = true;
                working_set.push_front(basic_block(suiv,
                                                   trace,
                                                   flag,
                                                   prev,
                                                   ret_addr,
                                                   depth));
                ret_addr.push_front(retour);
              }
            }
          }
        }
      }

      if (!use_trace) // si on utilise pas trace
      {
        basic_ana::delete_list(liste_ptrop(),*trace);
      }
    }
    // else trace est vide
  }

  //--------------------------------------------------------------------------
  // efface toutes les oprandes contenues dans la liste de blocs de base l
  void clean_bb_list(liste_bb* l)
  {
    for (liste_bb::iterator p = l->begin(); p != l->end(); ++p)
    {
       liste_ptrop* todel = (*p).in;
       basic_ana::delete_list(liste_ptrop(),*todel);
    }
  }
  //--------------------------------------------------------------------------
  // Effectue une analyse de flux de donnes
  // starting_address : adresse de dpart
  // starting_operand : oprande de dpart
  // mode : TRACE_FORWARD ou TRACE_BACKWARD
  //
  // renvoie une liste de netnode contenant les informations  afficher
  // chaque netnode est cod de telle manire :
  // altval(-1) = le nombre de lignes du netnode
  // altval(x) pour x >-1 : x ieme adresse
  // supval(x,['0'..'4']) : reprsente les infos sous forme de chane de car.
  //                        de l'adresse numro x
  // valeur du tag :
  // '0' = DF_ALIAS
  // '1' = CF_CHG
  // '2' = DF_COPY
  // '3' = DF_DESTR
  // '4' = CF_USE


  liste_node* global_trace(ea_t starting_address,
                           df_op_t* starting_operand,
                           int mode)
  {
    typedef list<basic_block> liste_bb; // liste de blocs de base
    liste_bb working_set,               // liste des blocs de base  analyser
             history;                   // liste des blocs de base
                                        // dj analyss

    liste_node* gti = new liste_node; // Global Trace Information : rsultats

    liste_ptrop templ;
    templ.push_front(starting_operand);

    // cre le premier bloc de base  analyser
    // initialis  l'adresse de dpart
    // avec une liste contenant l'oprande de dpart
    // un flag  NEXT_JMP
    // une pile des adresses de retour vide
    // et une profondeur de 0
    working_set.push_front(basic_block(starting_address,
                           new liste_ptrop(templ),
                           NEXT_JMP,
                           BADADDR,
                           liste_ret(),
                           0));
    templ.clear();
    // nombre de blocs de base analyss
    unsigned long int nbb = 0;

    // tant qu'il reste des blocs  analyser
    // et qu'on ne dpasse pas le nombre max de blocs
    while( (!working_set.empty()) && (nbb < MAX_BASIC_BLOCK) )
    {
      nbb++;

      // cre les paramtres "OUT" ncessaires  l'appel
      // de basic_ana::trace_forward et basic_ana::trace_forward
      liste_ptrop* out;
      out = new liste_ptrop();
      list<ea_t> next;
      netnode res;
      res.create();

      //prend le premier bloc  analyser et le retire du working_set
      basic_block d = *working_set.begin();
      working_set.remove(d);

      // ajoute le bloc  l'historique
      history.push_front(d);

      // paramtres "IN" de trace_forward et trace_backward
      ea_t ea            = d.ea;
      liste_ptrop* in    = d.in;
      unsigned char flag = d.flag;
      liste_ret ret_addr = d.ret_addr;
      int depth          = d.depth;

      ea_t prev = d.prev_ea;


      // appel le niveau infrieur
      if (mode == TRACE_FORWARD)
        basic_ana::trace_forward(ea,
                                 *in,
                                 *out,
                                 next,
                                 res,
                                 ret_addr,
                                 flag,
                                 prev,
                                 depth);
      else
        basic_ana::trace_backward(ea,
                                  *in,
                                  *out,
                                  next,
                                  res,
                                  ret_addr,
                                  flag,
                                  prev,
                                  depth);

      //-----------------------------------------------
      // traitement du rsultat
      //-----------------------------------------------

      // ajoute le resultat du bloc analys  la liste des rsultats
      // si le rsultat n'existe pas dj
      // sinon detruit res

      add_if_unique(res, gti);

      //-----------------------------------------------
      // traitement de la suite de l'analyse
      //-----------------------------------------------
      // ajoute  working set les blocs de base suivants non encore analyss
      feed_working_set(out,
                       next,
                       history,
                       working_set,
                       flag,
                       ret_addr,
                       mode,
                       prev,
                       depth );
    }

    // si l'analyse s'est arrte d au nombre de blocs de base analyss
    if (!working_set.empty())
    {
      msg("maximum number of basic block reached\n\
           Working Set size = %d\n", working_set.size());
      // efface les oprandes du working_set
      clean_bb_list(&working_set);
    }
    //else working_set vide

    // dsalloue la mmoire

    msg("delete history %d\n",history.size());

    clean_bb_list(&history);

    return gti;
  }

} // namespace