/*
Copyright (c) 2003, Michel Jean-Franois <jfmichel(at)operamail(dot)com>
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer. 

- 	Redistributions in binary form must reproduce the above copyright 
	notice, this list of conditions and the following disclaimer in the 
	documentation and/or other materials provided with the distribution. 

-	Neither the name of Sobek nor the names of its contributors 
	may be used to endorse or promote products derived from this software 
	without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#include "basic_ana.hpp"

#include <allins.hpp>
#include <bytes.hpp>
#include <frame.hpp>
#include <auto.hpp>
#include <typeinf.hpp>
#include <string>

// flags de prdictions
// force un valeur de flag

#define FORCE_NONE        0
#define FORCE_RET         1
#define FORCE_FALSE_CALL  2

namespace basic_ana
{
  //--------------------------------------------------------------------------
  // options d'analyse

  static uint options = 0;

  //--------------------------------------------------------------------------
  // tablit une option
  void set_option(uint flag)
  {
    options |= flag;
  }
  //--------------------------------------------------------------------------
  // teste une option
  bool test_option(uint flag)
  {
    return (options & flag) != 0;
  }
  //--------------------------------------------------------------------------
  // ajoute a 'l' les oprandes qui seront dtruites suite
  // a la destruction de l'oprande passe en paramtre
  void add_to_dependence_list(df_op_t* op, liste_ptrop& l)
  {
    // si l'op n'est pas dj dans la liste
    if ( find(l.begin(), l.end(), op) == l.end() )
      l.push_front(op);       // ajoute l'op  la liste

    if (op->aliases != NULL)  // si l'op a des alias
    {
      // appel rcursif
      liste_ptrop::iterator i_rem;
      for (i_rem = op->aliases->begin(); i_rem != op->aliases->end(); ++i_rem )
        add_to_dependence_list(*i_rem,l);
    }
  }
  //--------------------------------------------------------------------------
  // retire les lments de 'l' de la liste 'out' et efface le contenu de 'l'
  // retire les contraintes d'alias
  void delete_list(liste_ptrop &out, liste_ptrop& l)
  {

    // enlve les contraintes
    for (liste_ptrop::iterator p = l.begin(); p != l.end(); ++p)
    {
      df_op_t *op = *p;

      if (op->aliased != NULL )               // si this est un alias
      {
        if (op->aliased->aliases != NULL)
          op->aliased->remAlias(op);   // enleve la rfrence
        op->aliased = NULL;
      }
    }

    // remove
    for (liste_ptrop::iterator p = l.begin(); p != l.end(); ++p)
    {
      out.remove(*p);
      delete (*p);
    }
    l.clear();
  }
  //--------------------------------------------------------------------------
  // renvoie true si 'ea' est la limite d'un bloc de base
  // backward position a vrai indique un tracage arriere
  bool is_basic_block_end(ea_t genEA, bool backward)
  {
    ua_ana0(genEA);
    ea_t item_end = get_item_end(genEA);
    switch ( ph.notify(ph.is_basic_block_end) )
    {
      case 1:
      default:
        #ifdef SDK_46
        if ( is_call_insn(genEA) ) return true;  // call
        #else
        if ( is_call(cmd.itype) ) return true;  // call
        #endif
        // rupture du flux de controle en traage avant
        if ( get_first_fcref_from(genEA) != BADADDR && !backward) return true;
        // rupture du flux de controle en traage arriere
        if ( get_first_fcref_to(genEA) != BADADDR && backward) return true;
        // arrive t-on a lins suivant ea par un flux normal ?
        return !isFlow(getFlags(item_end));
      case 0:
        return false;
      case 2:
        return true;
    }
  }
  //--------------------------------------------------------------------------
  // renvoie l'adresse suivante en ea rompant le flux de contrle

  ea_t get_jump_target(ea_t ea)
  {
    // we pass the execution somewhere
    ea_t jump = get_first_fcref_from(ea);
    if ( jump == BADADDR )
      return BADADDR;

    // only one target
    if ( get_next_fcref_from(ea, jump) != BADADDR )
      return BADADDR;

    // we don't pass execution to the next instruction
    ea_t next = get_item_end(ea);
    flags_t F = getFlags(next);
    if ( isFlow(F) )
      return BADADDR;

    return jump;
  }
  //--------------------------------------------------------------------------
  // ajuste les oprandes de pile de out
  // delta : dplacement a ajouter aux oprandes
  // limit : si le dplacement d'une oprande ajuste
  //         est plus petite que limit, elle est dtruite

  void adjust_operands(liste_ptrop &out, long delta, long limit)
  {
    liste_ptrop post_remove;
    for(liste_ptrop::iterator p = out.begin(); p != out.end(); ++p)
    {
      df_op_t *op = *p;
      if (op->norm)
      {
        op->addr += delta;

        if ( (signed long)(op->addr) < (signed long)(limit) )
        {
          add_to_dependence_list(op,post_remove);
        }
      }
    }
    delete_list(out, post_remove);
    out.delta_total += delta;
  }
  //--------------------------------------------------------------------------
  // affiche une ligne dans la fentre des messages
  // cette ligne  la structure suivante :
  // [ea]:[format][liste]
  //
  // ea : une adresse
  // format : chane de caractres formates
  // list : liste de pointeurs d'oprandes

  void print_list_ptrop(const liste_ptrop &list,
                               ea_t ea,
                               const char *format, ...)
  {
    char buf[MAXSTR];
    a2funcoff(ea, buf, MAXSTR);
    msg("%s: ", buf);

    va_list va;
    va_start(va, format);
    vmsg(format, va);
    va_end(va);
    for(liste_ptrop::const_iterator p = list.begin(); p != list.end(); ++p)
    {
      df_op_t *op = *p;
      msg(" %s", op->getTxt(list.delta_total));
    }
    msg("\n");
  }

  //--------------------------------------------------------------------------
  // renvoie la fonction appele en ea
  // si la fonction est de type 'Thunk' renvoie la fonction reelement appele

  func_t *get_called_non_thunk_function(ea_t ea)
  {
    ea_t callee = get_first_fcref_from(ea);
    func_t *cfn = get_func(callee);

    // if normal functions + jump functions...
    if ( cfn != NULL && (cfn->flags & FUNC_THUNK) == 0 // not thunk
      || get_jump_target(callee) != BADADDR ) // jump func
    {
      // passer les thunk call
      // on doit passer les THUNK FUNC pour arriver a la "dernire"
      while( true )
      {
        ea_t jump = get_jump_target(callee);
        if ( jump == BADADDR )
          break;
        callee = jump;
      }
      return get_func(callee);
    }
    return NULL;
  }

  //--------------------------------------------------------------------------
  // dtruit les oprandes de la liste 'out'
  // n'tant pas des paramtres de la fonction en 'ea'
  void destroy_non_param_regs(ea_t ea, liste_ptrop &out)
  {
    type_t buff[MAXSTR];
    type_t* ptrbuff = buff;
    liste_ptrop post_remove;

    //liste_ptrop regs_to_del ;
    list<RegNo> regs_to_del ;

    // cre la liste des registres eax,ecx,edx
    // (edi esi ebx et ebp sont toujours sauvs voire hide_op unhide_op)
    // ces registres seront dtruits a moins qu'il soient utiliss
    // par la convention d appel de la prochaine fonction

    regs_to_del.push_front(R_ax);
    regs_to_del.push_front(R_cx);
    regs_to_del.push_front(R_dx);

    if(get_ti(ea,buff,NULL))
    {
      ulong arglocs[MAX_FUNC_ARGS];
      int n = calc_arglocs(ptrbuff,arglocs,qnumber(arglocs));

      for (int i = 0; i < n; ++i)
      {
        if(arglocs[i] & ARGLOC_REG)
          // retire les registres utiliss par la convention d'appel
          regs_to_del.remove((RegNo)(arglocs[i] & !ARGLOC_REG));
      }
    }

    // regs_to_del contient la liste des registres  effacer

    for(list<RegNo>::iterator r = regs_to_del.begin();
        r != regs_to_del.end();
        ++r)
    {
      df_op_t* cop = new df_op_t(ea,*r,dt_dword);

      for(liste_ptrop::iterator p = out.begin(); p != out.end(); ++p)
      {

        if (*cop <= **p) //dtruit toutes les oprandes utilisant ces registres
        {
            add_to_dependence_list(*p,post_remove);
        }
      }
      delete cop;
    }

    delete_list (out,post_remove);
  }
  //--------------------------------------------------------------------------
  // cache les op. de 'l'
  // qui ne sont pas les paramtres de la prochaine fonction
  // ces oprandes sont :
  // les op. normalises dont l addr est suprieur a zro (variables locales)
  // les registres esi edi ebx et ebp qui ne sont jamais utiliss
  // par les conventions d appel
  // depth : profondeur actuelle

  void hide_op(liste_ptrop l, int depth)
  {
    for(liste_ptrop::iterator p = l.begin(); p != l.end();++p)
    {
      // caches les operandes de pile
      if ((*p)->norm && !(*p)->hide && (signed long)((*p)->addr) > 0)
      {
        (*p)->hide = true ;
        (*p)->hide_depth = depth ;
      }
      else
      {
        short rnum = (*p)->getReg();
        if ( (rnum != -1))
        { // caches le registres di si bx et bp
          if (rnum == R_di || rnum == R_si || rnum == R_bx || rnum == R_bp)
          {
            (*p)->hide = true ;
            (*p)->hide_depth = depth ;
          }

        }
      }

    }
  }
  //--------------------------------------------------------------------------
  // rhabilite les oprandes caches de l
  // depth : profondeur actuelle
  // (voir hide_op)

  void unhide_op(liste_ptrop l, int depth)
  {
    for(liste_ptrop::iterator p = l.begin(); p != l.end();++p)
    { // rhabilite les oprande de pile
      if (    (*p)->norm
           && (*p)->hide
           && (signed long)((*p)->addr) > 0
           && (*p)->hide_depth == depth
         )
      {
        (*p)->hide = false ;
      }
      else if (depth == (*p)->hide_depth)
      {
        short rnum = (*p)->getReg();
        if ( (rnum != -1))
        { // rehabilite les registres di si bx et bp
          if (rnum == R_di || rnum == R_si || rnum == R_bx || rnum == R_bp)
            (*p)->hide = false ;
        }
      }

    }
  }

  //--------------------------------------------------------------------------
  // ajustement des oprandes en traage avant
  // ea : adresse courante
  // flag : indique comment le contrle de flux a t rompu
  // out : liste contenant les oprandes  ajuster
  // callstate : pile des adresses de retour
  // prev_ea : une adresse du bloc de base prcdant
  //           cette valeur n'a de sens que si flag est a NEXT_RET
  //           si le bloc de base prcdant est une fonction Thunk,
  //           prev_ea vaut BADADDR
  // depth : profondeur actuelle

  void adjust_function_operands_forward(const ea_t ea,
                                        const uchar &flag,
                                              liste_ptrop &out,
                                        const liste_ret &callstate,
                                              ea_t prev_ea,
                                              int depth)
  {
    func_t *pfn = get_func(ea);

    if (pfn != NULL)
    {
      long delta = 0;
      long limit = INT_MIN;

      if (flag == NEXT_CALL)
      {
        if ((pfn->flags & FUNC_THUNK) == 0)
        {
          // caller
          ea_t caller = decode_prev_insn(callstate.front().ea);
          func_t *pfn_caller = get_func(caller);
          if (pfn_caller != NULL)
          {
            long spd_caller = get_spd(pfn_caller,caller);
            // callee
            ulong frame_size = get_frame_size(pfn);

            // calcul de l'ajustement
            delta = frame_size - spd_caller - pfn_caller->frregs
                    - pfn_caller->frsize - pfn->argsize ;

            // cache les oprandes
            if (test_option(HIDE_NON_ARG_OP_IN_SUB))
              hide_op(out,depth);

            // dtruit les oprandes qui ne sont pas des paramtres
            if (test_option(USE_CALLING_CONVENTION))
             destroy_non_param_regs(ea,out);
          }
        }
      }
      else if (flag == NEXT_RET)
      {
        ea_t prev_insn = decode_prev_insn(ea);
        func_t *cfn ;
        if (prev_ea != BADADDR)
          cfn = get_func(prev_ea); // fonction nomale
        else
          cfn = get_called_non_thunk_function(prev_insn); // fonction thunk

        if ( cfn != NULL )
        {
          int callee_frame_size = get_frame_size(cfn);
          int caller_spd        = get_spd(pfn,prev_insn);
          int callee_argsize    = cfn->argsize;

          // calcul de l'ajustement
          delta = caller_spd - callee_frame_size + callee_argsize
                  + pfn->frregs + pfn->frsize;
          limit = get_spd(pfn, ea) + pfn->frregs + pfn->frsize;
        }
       }

      // ajuste les oprandes
      adjust_operands(out, delta, limit);

      if ( test_option(HIDE_NON_ARG_OP_IN_SUB) && flag == NEXT_RET)
        unhide_op(out,depth);
    }
  }
  //--------------------------------------------------------------------------
  // charge les oprandes de l'instruction en ea dans le tableau d'oprandes op
  // Renvoie le nombre d'oprandes de l'instruction

  int loadOperands(/*IN*/ea_t ea, /*OUT*/df_op_t** op)
  {
    func_t* f = get_func(ea);

    // nombre d'oprandes
    ushort nb_op = get_nb_op();

    int cur_op;
    for (cur_op = 0; cur_op < nb_op; ++cur_op)
    {
      op[cur_op] = new df_op_t(cur_op);
      op[cur_op]->normalize();
    }

    // l'instruction utilise une oprande sur la pile ?
    if (cmd.itype == NN_push || cmd.itype == NN_pop)
    {
      // si c un push le spd  prendre en compte
      // est celui de l'instruction suivante
      ulong offset = cmd.itype == NN_push
                   ? get_sp_delta(f,ea + cmd.size)
                   : 0; // sinon c un pop

      op[cur_op] = op[0]->create_normalized_operand(offset);

      ++cur_op;
    }
       // correction des instruction ayant besoin d'oprandes supplmentaires
       // ex: mul imul div idiv
    else if ( (cmd.itype == NN_mul)
           || (cmd.itype == NN_div)
           || (cmd.itype == NN_idiv)
            )
    {
      switch ( op[1]->dtyp )
      {
        case dt_word:
          op[0]->dtyp = dt_word;
        case dt_dword:
          op[2] = new df_op_t(ea, R_dx, op[1]->dtyp);
          op[2]->n = 2 ;
          ++cur_op;
          break;
        case dt_byte:
          op[0]->dtyp = dt_word;
          break;
      }
    }
    else if ( cmd.itype == NN_imul)
    {
      if (cur_op == 1)
      {
        if(!(op[0]->showed())) // forme  une oprande explicite
        {
          switch ( op[1]->dtyp )
          {
            case dt_word:
              op[0]->dtyp = dt_word;
            case dt_dword:
              op[2] = new df_op_t(); // op vide
              op[3] = new df_op_t(ea, R_dx, op[1]->dtyp);
              op[3]->n = 3 ;
              ++cur_op;
              break;
            case dt_byte:
              op[0]->dtyp = dt_word;
              break;
          }
        }
        //else                  // forme  deux oprandes explicites
        //{
        //} // ne ncessite aucune modification
      }
      else if (cur_op == 2)   // forme  trois oprandes explicites
      {
        op[3] = new df_op_t(*op[0]); // duplique l'op 0
        op[0]->type = o_void ; // et dsactive l'op 0

        ++cur_op;
      }
    }

    return cur_op;
  }
  //--------------------------------------------------------------------------
  // dtruit l'oprande op si elle n'est plus valide en ea
  // ea : adresse courante
  // pfn : fonction en ea
  // op : oprande  tester


  void mark_dead_operand(ea_t ea, func_t* pfn, df_op_t* op)
  {
    liste_ptrop rem;

    if ( (pfn != NULL)
      && (ea < pfn->endEA)
      && !(pfn->flags & FUNC_THUNK)
      && (op->norm == true)
      && (int(op->addr) < int(get_spd(pfn,ea)+pfn->frregs+pfn->frsize)) )
    {
      add_to_dependence_list(op,rem);
      // on met les ops a void pour ne pas qu'elle soit dtecte
      // elle seront effaces en fin de boucle
      for (liste_ptrop::iterator p = rem.begin(); p != rem.end(); ++p)
      {
        (*p)->type = o_void;
      }
    }
  }
  //--------------------------------------------------------------------------
  // effectue l'analyse spciale de l'instruction courante (cmd)
  // n : numro de l'oprande  analyser
  // op[] : tableau des oprandes de l'instruction
  // destr : renvoie true si l'oprande n est dtruite
  // xch1 :  renvoie true si n==1 et si l'instruction est Xchg
  // xch2 :  renvoie true si n==2 et si l'instruction est Xchg

  void special_ana(int n, df_op_t *op[], bool &destr, bool &xch1, bool &xch2)
  {
    if (!n) // si on analyse la premire oprande
    {
      // Cas spciaux o l'oprande courante doit tre dtruite

      //      sub/sbb/xor opx,opx
      if ( ( (cmd.itype == NN_xor)
          || (cmd.itype == NN_sub)
          || (cmd.itype == NN_sbb)) && (*op[0] == *op[1]) )
      {
        destr = true;
      }
      else
      //      or opx,-1
      if ( (cmd.itype == NN_or)
        && ( ((*op[1]).type == o_imm) && ((*op[1]).value == -1) ) )
      {
        destr = true;
      }
      else
      //      and opx,0
      if ( (cmd.itype == NN_and)
        && ( ((*op[1]).type == o_imm) && ((*op[1]).value == 0) ) )
      {
        destr = true;
      }
      if (cmd.itype == NN_xchg)
      {
        xch1 = true;
      }
    }
    else
    {
      if (cmd.itype == NN_xchg)
      {
        xch2 = true;
      }
    }
  }
  //--------------------------------------------------------------------------
  // renvoie la prochaine instruction dlimitant un bloc de base
  // ea : instruction courante
  // mode : true traage arrire
  //        false traage avant

  ea_t reach_end_of_block(ea_t ea, const bool mode = false)
  {
    bool basic_block_end  = is_basic_block_end(ea,mode) ;

    while (!basic_block_end)
    {
      if (mode == true)
        ea = decode_prev_insn(ea);
      else
        ea += cmd.size;
      ua_ana0(ea);
      showAddr(ea);
      if (is_basic_block_end(ea,mode))
        basic_block_end = true;
    }

    return (ea);
  }

  //--------------------------------------------------------------------------
  // cre le rsultat de l'analyse de l'instruction courante
  // event[] tableau des vnements
  // res : netnode associe au bloc de base courant
  // count : numro de l'instruction courante par rapport au bloc de base
  //
  // res est cod de telle manire :
  // altval(-1) = le nombre de lignes du netnode
  // altval(x) pour x >-1 : adresse numro x
  // supval(x,['0'..'4']) : reprsente les infos sous forme de chane de car.
  //                        de l'adresse numro x
  // valeur du tag :
  // '0' = DF_ALIAS
  // '1' = CF_CHG
  // '2' = DF_COPY
  // '3' = DF_DESTR
  // '4' = CF_USE
  //-------------------------------------------------------------------------

  void create_result(liste_op event[], netnode res, int count)
  {
    for (int i = 0; i < 5; ++i)
    {
      string txt;
      for (liste_op::iterator i_event = event[i].begin();
          i_event != event[i].end();
          ++i_event)
      {
      	txt += (*i_event).getTxt();
      	txt += " ";
      }

      if (!txt.empty())
        res.supset(count-1,txt.c_str(), 0, char(i+'0'));
    }
  }

  //--------------------------------------------------------------------------
  // dtermine les paramtres des blocs de base suivants en traage avant
  // ea : fin du bloc de base courant
  // vide : indique si la liste des oprandes  tracer est vide ou
  //        ne contient que des oprandes caches
  //
  // met  jour les paramtres
  //
  // depth : profondeur
  // callstate : pile des adresses de retour
  // next : liste des adresses des blocs de base suivants
  // flag : indique la manire dont le flux de contrle est rompu
  // prev_ea : adresse du bloc de base courant si le contrle
  //           de flux est rompu par un retour de fonction

  void link_forward(ea_t ea,
                    int &depth,
                    liste_ret &callstate,
                    list<ea_t> &next,
                    uchar &flag,
                    bool vide,
                    ea_t& prev_ea)
  {

    // force un next_false_call ou un mext_ret si vide est vrai

    uchar force = FORCE_NONE;
    if (vide && test_option(HIDE_NON_ARG_OP_IN_SUB))
    {
      if(Ins_Op_Test(CF_CALL)) // call
      {
        // si la liste ne contient que des oprandes caches
        // on force un FALSE_CALL
        force = FORCE_FALSE_CALL;
      }
      else if (Ins_Op_Test(CF_STOP) && (get_first_cref_from(ea) == BADADDR))
      {
        force = FORCE_NONE; // si on est sur un ret on ne force rien
      }
      else
      {
        force = FORCE_RET; // sinon on est en plein dans une fonction
                           // on force un retour de fonction
      }

    }

    // traitement principal

    if (Ins_Op_Test(CF_CALL) && (force != FORCE_RET)) // CALL
    {
      func_t* f = get_func(ea);

      flag = NEXT_CALL;

      if ( (get_first_fcref_from(ea) != BADADDR) && (force != FORCE_FALSE_CALL))
      { // le call pointe sur du code valide
        ret addret;

        addret.depth = depth;
        ++depth; // on entre ds un call

        // pour CALL du genre "call [base_table+index*addrsize]"
        for ( ea_t code_ref = get_first_fcref_from(ea);
              code_ref != BADADDR;
              code_ref = get_next_fcref_from(ea,code_ref) )
        {
          // met l'adresse des fonctions appeles par le call ds next
          next.push_front(code_ref);
        }

        if (ea+cmd.size < f->endEA)
          addret.ea = ea+cmd.size;
        else     // un call est la dernire ins d'une fonction ?
          addret.ea = BADADDR;

        list<ea_t> temp_ret;
        getRetFromFunc(get_first_fcref_from(ea), temp_ret);
        int nb_ret = temp_ret.size();

        if (nb_ret) // si on a au moins un ret
        {
          // met  jour la pile des adresses de retour
          callstate.push_front(addret);
        }
        else // la fonction n a pas de ret !
        {
          func_t* fc = get_func(get_first_fcref_from(ea));

          // si fonction de type Thunk, c'est normal
          if (fc && fc->flags & FUNC_THUNK)
          {
            callstate.push_front(addret);
          }
          //else
          //{
          // on rentre ds un call qui n a pas de ret
          //}
        }
      }
      else // impossible de parcourir la fonction appele (ex call eax,...  )
      {
        flag = NEXT_FALSE_CALL;
        if (ea+cmd.size != f->endEA)
          next.push_front(ea+cmd.size);
        //else
          // pas d'instruction suivant le call
      }
    }
    else if ( (force == FORCE_RET)
           || (Ins_Op_Test(CF_STOP)
           && (get_first_cref_from(ea) == BADADDR)) )
    // ret ou jump de thunk call vers import table
    {
      flag = NEXT_RET;

      func_t* f = get_func(ea);

      if (f && (f->flags & FUNC_THUNK))
        prev_ea = BADADDR;
      else
        prev_ea = ea;

      if (!callstate.empty())
      {
        if (callstate.front().ea != BADADDR)
        {
          depth = callstate.front().depth;

          next.push_front(callstate.front().ea);
          callstate.pop_front();
        }
        else
        {
          callstate.pop_front();
        }
      }
      else // callstate.empty()
      {
        depth--;
        // la pile des adresses de retour est vide
        // next reoit l'ensemble des adresses de retour possibles
        getRetAddr(ea,next);
      }
    }
    else  // jmp/Jcxx
    {
      flag = NEXT_JMP;
      // un jmp dans un thunk est considr comme un call
      //(afin que les ops  normaliser soient traites dans la "vrai" fonction)
      if (get_func(ea) && get_func(ea)->flags & FUNC_THUNK)
      {
        flag = NEXT_CALL;
        // callstate a dj t ajuste lors du call du THUNK
      }

      // blocs de base suivants
      for ( ea_t code_ref = get_first_cref_from(ea);
            code_ref != BADADDR;
            code_ref = get_next_cref_from(ea,code_ref) )
      {
        next.push_front(code_ref);
      }
    }

  }
  //--------------------------------------------------------------------------
  // effectue l'analyse de flux de donnes en traage avant d'un bloc de base
  //
  // ea  : adresse du bloc de base
  // in  : liste des oprandes vivantes en ea
  // out : liste des oprandes vivantes a la fin du bloc de base
  // next: liste des adresses des blocs de base suivants
  // result : rsultat de l'analyse du bloc courant
  // callstate : pile des adresses de retour associe au bloc de base courant
  // flag : indique la manire dont le flux de contrle est rompu
  // prev_ea : adresse du bloc de base prcdant (entre) et courant (sortie)
  //           si le contrle de flux est rompu par un retour de fonction
  // depth : profondeur du bloc courant

  void trace_forward( /*IN*/ ea_t ea,
                      /*IN*/ const liste_ptrop &in,
                      /*OUT*/ liste_ptrop &out,
                      /*OUT*/ list<ea_t> &next,
                      /*OUT*/ netnode result,
                      /*IN/OUT*/liste_ret &callstate,
                      /*IN/OUT*/ unsigned char &flag,
                      /*IN/OUT*/ ea_t& prev_ea,
                      /*IN/OUT*/int &depth)
  {

    bool basic_block_end = false;
    liste_ptrop post_remove, post_add, potential_remove;
    int counter = 0;

    duplicate_and_append(in,out); // duplique in dans out

    #ifdef DEBUG_LIST
    print_list_ptrop(out, ea, "trace_forward in:");
    #endif

    // ajustement
    adjust_function_operands_forward(ea, flag, out, callstate, prev_ea, depth);

    //-----------------------------------------------
    // Traitement principal
    //-----------------------------------------------

    // affiche l'adresse courante dans l'interface de ida
    callui(ui_setstate,st_Work);
    ua_ana0(ea);
    showAddr(ea);

    while (!basic_block_end && !out.vide())
    {
      liste_op event[5]; // vnements

      // 6 oprandes
      df_op_t *op[UA_MAXOP] = {NULL, NULL, NULL, NULL, NULL, NULL};

      // indique les oprandes inutiles (true)
      bool del_op[UA_MAXOP] = {true, true, true, true, true, true};

      bool xchg_1 = false, xchg_2 = false;
      bool dif_type_to_eq = false;

      // nombre d'oprandes effectivement utilises
      ushort nb_op = loadOperands(ea, op);  // charge les oprandes

      if (nb_op) // si l'ins courante a des oprandes
      {

        // trouve les lments de out correspondant aux oprandes
        for (int i=0; i < nb_op; ++i)  // parcours des oprandes
        {
          // parcours de la liste  tracer
          for (liste_ptrop::iterator i_out = out.begin();
               i_out != out.end();
               ++i_out)
          {
            bool destroyed = false;
            bool copied = false;
            bool block_aliased = false;
            bool used = false;

            df_op_t *cop = *i_out;

            // dtection des oprandes de pile  effacer

            mark_dead_operand(ea, get_func(ea), cop);

            // si l'lment courant de out est comparable l'oprande courante

            if (     (*cop <= (*op[i]) )
                || ( (*cop >= (*op[i])) && Op_Is_Destroyed(i+1) )
               )
            {

              // dtecte certains cas particuliers
              special_ana(i, op, destroyed, xchg_1, xchg_2);

              //-----------------------------------------------
              // si les types sont diffrents
              // => les deux ops sont comparables mais pas identiques
              //-----------------------------------------------

              if ( cop->type != op[i]->type )
              {
                block_aliased = true;
                // empche les alias si type !=
                // pour considrer qu'il y a un alias il faut que
                // l'op courante soit identique a l'op trace

                if (cop->type == o_reg)
                {
                  list<RegNo> l_int;
                  int c = op[i]->getRegsFromPhrase(l_int);

                  if ( (c == 1) && ( *(l_int.begin()) == cop->reg ) )
                  { // si cop est un registre
                    // et que op[i] est un mode d'adressage
                    // n'utilisant que ce registre
                    dif_type_to_eq = true;

                    if (Op_Is_Aliased(i+1)) // ex : lea eax,[ebx]
                    {
                      copied = true; // force la copie
                    }
                  }
                  else
                    used = true; // force l'utilisation
                }

                // si les types sont diffrents et que l'op est dtruite
                // ex : mov [eax+edx],0
                // eax est utilis pas dtruite

                if (Op_Is_Destroyed(i+1))
                {

                  if (*cop <= (*op[i]))
                  {
                    used = true;      // cas cop : eax, op[i] : [eax+edx]
                  }
                  else
                  {
                    dif_type_to_eq = true;
                    destroyed = true;  // cas cop : [eax+edx], op[i] : eax
                  }
                }
              }

              // si l'op est une o_imm elle n'est intressante
              // a suivre que lors de sa premire utilisation
              // (ex : on veut suivre "push 3" on initialisera in
              // avec l'op imm 3)

              if ((*i_out)->type == o_imm)
              {
                post_remove.push_front(*i_out);
              }

              //-----------------------------------------------
              // si les types sont gaux
              //-----------------------------------------------

              if ( ((*i_out)->type == op[i]->type) || (dif_type_to_eq))
              {
                //-----------------------------------------------
                // Op_Is_Destroyed
                //-----------------------------------------------

                if ( (Op_Is_Destroyed(i+1) || destroyed))
                {
                  // doit-on afficher les destructions des oprandes ?
                  if (basic_ana::test_option(SHOW_DESTROY))
                  {
                    // si l'op n'est pas dj dans l'vnement
                    if ( find(event[3].begin(), event[3].end(), *op[i])
                         == event[3].end() )
                      event[3].push_front(*op[i]);
                  }

                  // ajoute l'op. courr.  la liste des lments
                  //  retirer de out
                  potential_remove.push_front(*i_out);
                }

                //-----------------------------------------------
                // Op_Is_Copied
                //-----------------------------------------------

                if (Op_Is_Copied(i+1) || copied)
                {

                  // si l'op n'est pas dj dans l'vnement
                  if ( find(event[2].begin(), event[2].end(), *op[i])
                       == event[2].end() )
                  {
                    event[2].push_front(*op[i]);
                  }

                  // un oprande 'copied' se copie dans les oprandes 'changed'
                  for (int j=0; j < nb_op; ++j)
                  {
                    if (Op_Is_Changed(j+1)  && i!=j) // i!=j pour cas du Xchg
                    {
                      // si cop est l'alias d'une oprande
                      if (cop->aliased)
                      {
                        op[j]->aliased = cop->aliased;

                        // ajoute l'op courante
                        //  la liste des alias de l'op aliase
                        if(op[j]->aliased->aliases)
                        	op[j]->aliased->aliases->push_front(op[j]);
                      }

                      // si op[j] n'est ni dj dans out ni dans post_add
                      if(   (find(post_add.begin(), post_add.end(), op[j])
                             == post_add.end())
                         && (find(out.begin(), out.end(), op[j]) == out.end())
                         )
                      {

                        post_add.push_front(op[j]);
                        del_op[j] = false;
                      }
                    }
                  }
                }

                //-----------------------------------------------
                // Op_Is_Aliased
                //-----------------------------------------------

                if (Op_Is_Aliased(i+1) && !block_aliased)
                {
                  // si l'op n'est pas dj dans l'vnement
                  if ( find(event[0].begin(), event[0].end(), *op[i])
                       == event[0].end() )
                  {
                    event[0].push_front(*op[i]);
                  }

                  // un oprande 'aliased' se copie dans les oprandes 'changed'
                  for (int j=0; j < nb_op; ++j)
                  {
                    if (Op_Is_Changed(j+1)  && i!=j) // i!=j pour cas du Xchg
                    {

                      df_op_t *alias_op = op[j] ;

                      // ajouter l alias  la liste
                      cop->addAlias(alias_op);

                      // l alias obtient une rfrence  l'oprande
                      alias_op->setAliased(cop);

                      // si alias_op n'est pas dj dans post_add et out
                      if(   (find(post_add.begin(), post_add.end(), alias_op)
                             == post_add.end())
                         && (find(out.begin(), out.end(), alias_op)
                             == out.end()))
                      {
                        post_add.push_front(alias_op); // ajouter l'alias
                        del_op[j] = false;
                      }
                    }
                  }
                }

                //-----------------------------------------------
                // Op_Is_Changed
                //-----------------------------------------------

                if (Op_Is_Changed(i+1))
                {

                  if (!Op_Is_Destroyed(i+1)) // changed < destroyed
                  {
                    // si l'op n'est pas dj dans l'vnement
                    if ( find(event[1].begin(), event[1].end(), *op[i])
                         == event[1].end() )
                    {
                      event[1].push_front(*op[i]);
                    }
                  }
                }
              }
              //-----------------------------------------------
              // Op_Is_Used
              //-----------------------------------------------

              if (Op_Is_Used(i+1) || used)
              {
                // si l'op n'est pas dj dans l'vnement
                if ( find(event[4].begin(), event[4].end(), *op[i])
                     == event[4].end() )
                {
                  event[4].push_front(*op[i]);
                }
              }

              if ( !(   event[0].empty()
                     && event[1].empty()
                     && event[2].empty()
                     && event[3].empty()
                     && event[4].empty()) )
              // ne marquer l'adresse courrante
              // que si au moins un vnement a t dtect
                if (result.altval(counter-1) != ea)
                // ne pas mettre deux fois la meme adresse dans la liste
                  result.altset(counter++,ea);
                  // ajoute l'adresse dans le rsultat
            }
          }

        }
      }

      // ajoute les oprandes marques  post_remove
      for(liste_ptrop::iterator p = out.begin(); p != out.end(); ++p)
      {
        if ((*p)->type == o_void)
        {
          post_remove.push_front(*p);
        }
      }

      if (xchg_1 && xchg_2)
        potential_remove.clear();  // ne pas effacer si on swap les oprandes
      else
        post_remove.splice(post_remove.end(), potential_remove);

      // post_remove contient la liste des ops  effacer

      liste_ptrop templ;

      for (liste_ptrop::iterator p = post_remove.begin();
           p!= post_remove.end();
           ++p)
      {
        basic_ana::add_to_dependence_list(*p,templ);
      }
      // templ contient la liste tendue des ops  effacer

      // efface et retire de out les oprandes dtruites
      delete_list(out, templ);
      post_remove.clear();

      // ajoute  out les copie et alias cres par l'instruction courante
      // nb : l'ajout est fait APRES le retrait.
      out.splice(out.end(), post_add);

      //-----------------------------------------------
      // commentaires
      //-----------------------------------------------

      create_result(event, result, counter);

      //-----------------------------------------------
      // incrmentation et teste de la fin du bloc de base
      //-----------------------------------------------

      if (is_basic_block_end(ea))
        basic_block_end = true;
      else
      {
        ea += cmd.size;
        ua_ana0(ea);
        showAddr(ea);
      }

      //-----------------------------------------------
      // dtruit les oprandes non utilises
      //-----------------------------------------------

      for (int i = 0; i < nb_op; ++i)
      {
     	  if (del_op[i])
     	    delete op[i];
      }

    }
    result.altset(-1,counter);

    // positionne ea sur la fin du bloc de base courant
    // si ce n'est pas dj le cas (out vide)
    ea = reach_end_of_block(ea);

    //-----------------------------------------------
    // Liens entre les blocs de base
    // ____________________________
    //
    // rempli et positionne depth, next, flag, prev_ea et callstate
    //-----------------------------------------------

    link_forward(ea, depth, callstate, next, flag,out.vide(), prev_ea);

    #ifdef DEBUG_LIST
    print_list_ptrop(out, ea, "trace_forward out:");
    #endif
  }

  //--------------------------------------------------------------------------
  // ajustement des oprandes en traage arrire
  // ea : adresse courante
  // flag : indique comment le contrle de flux a t rompu
  // out : liste contenant les oprandes a ajuster
  // callstate : pile des adresses de retour
  // prev_ea : une adresse du bloc de base prcdent
  // depth : profondeur actuelle

  void adjust_function_operands_backward(const ea_t ea,
                                             const uchar &flag,
                                             liste_ptrop &out,
                                             const ea_t prev_ea,
                                             int depth)
  {
    liste_ptrop post_remove;

    func_t *pfn = get_func(ea);
    func_t *prev_func = get_func(prev_ea);
    long delta = 0, limit = INT_MIN;

    if (pfn && prev_func)
    {
      if (flag == NEXT_CALL)
      {
        long  spd = get_spd(prev_func,prev_ea);
        ulong frsize = get_frame_size(pfn);

        delta = frsize - spd - prev_func->frregs
                - prev_func->frsize - pfn->argsize;

        if (test_option(HIDE_NON_ARG_OP_IN_SUB))
          hide_op(out,depth);

      }
      else if (flag == NEXT_RET)
      {

        ua_ana0(ea);
        delta =    get_spd(pfn,ea)
                -  get_frame_size(prev_func)
                +  prev_func->argsize
                + pfn->frregs
                + pfn->frsize;

        limit = get_spd(pfn, ea) + pfn->frregs + pfn->frsize;
      }

      adjust_operands(out, delta, limit);

      if ( test_option(HIDE_NON_ARG_OP_IN_SUB) && flag == NEXT_RET)
        unhide_op(out,depth);
    }
  }

  //--------------------------------------------------------------------------
  // dtermine les paramtres des blocs de base suivants en traage arrire
  // ea : fin du bloc de base courant
  //
  // met  jour les paramtres :
  //
  // depth : profondeur
  // callstate : pile des adresses de retour
  // next : liste des adresses des blocs de base suivant
  // prev_ea : adresse du bloc de base courant

  void link_backward(ea_t ea,
                     int &depth,
                     liste_ret &callstate,
                     list<ea_t> &next,
                     uchar &flag,
                     ea_t &prev_ea)
  {
    func_t* f;

    if (Ins_Op_Test(CF_CALL) && (flag != NEXT_RET) ) // call
    {
      flag = NEXT_CALL;

      f = get_func(get_first_fcref_from(ea));

      if ( (get_first_fcref_from(ea) != BADADDR) && f)
      // le call branche sur une adresse valide qui est dans une fonction
      {
        // ajoute l'adresse de retour
        callstate.push_front(ret(ea,depth));
        ++depth;

        // obtient la liste des adresses des ret des fonctions appeles

        for ( ea_t code_ref = get_first_fcref_from(ea);
              code_ref != BADADDR;
              code_ref = get_next_fcref_from(ea,code_ref) )
        {
          getRetFromFunc(code_ref,next);
        }
      }
      else
      // False Call impossible de tracer le call (ex Call eax)
      // on passe aux bloc de base suivants
      {
        flag = NEXT_FALSE_CALL;
        for ( ea_t code_ref= get_first_cref_to(ea);
              code_ref != BADADDR;
              code_ref = get_next_cref_to(ea,code_ref) )
        {
          next.push_front(code_ref);
        }
      }
    }
    else // pas de CALL
    {
      f = get_func(ea);
      if (f && ea == f->startEA) // dbut d'une fonction -> NEXT_RET
      {
        flag = NEXT_RET;

        if (!callstate.empty())
        {
          // branche sur l'adresse de retour
          next.push_front(callstate.front().ea);
          depth = callstate.front().depth;
          callstate.pop_front();
        }
        else // callstate vide -> branche sur toutes les adresses de retour
        {
          --depth;
          for ( ea_t code_ref= get_first_fcref_to(ea);
                code_ref != BADADDR;
                code_ref = get_next_fcref_to(ea,code_ref) )
          {
            next.push_front(code_ref);
          }
        }
      }
      else  // cas normal (jump,...)
      {
        flag = NEXT_JMP;
        for ( ea_t code_ref= get_first_cref_to(ea);
              code_ref != BADADDR;
              code_ref = get_next_cref_to(ea,code_ref) )
          next.push_front(code_ref);
      }
    }
    prev_ea = ea; // retient l adresse du bloc prcdant
  }

  //--------------------------------------------------------------------------
  // effectue l'analyse de flux de donnes en traage arrire d'un bloc de base
  //
  // ea  : adresse du bloc de base
  // in  : liste des oprandes vivantes en ea
  // out : liste des oprandes vivantes  la fin du bloc de base
  // next: liste des adresses des blocs de base suivants
  // result : rsultat de l'analyse du bloc courant
  // callstate : pile des adresses de retour associ au bloc de base courant
  // flag : indique la manire dont le flux de contrle est rompu
  // prev_ea : adresse du bloc de base prcdent (entre) et courant (sortie)
  // depth : profondeur du bloc courant

  void trace_backward(/*IN*/ ea_t ea,
                                 /*IN*/ const liste_ptrop& in,
                                 /*OUT*/ liste_ptrop& out,
                                 /*OUT*/ list<ea_t>& next,
                                 /*OUT*/ netnode result,
                                 /*IN/OUT*/liste_ret &callstate,
                                 /*IN/OUT*/unsigned char &flag,
                                 /*IN/OUT*/ea_t &prev_ea,
                                 /*IN/OUT*/int &depth)
  {
    bool not_basic_block_extremity = true, destroyed = false;
    bool ret_from_call = false ;
    liste_ptrop post_remove, post_add, potential_remove;
    int counter = 0;

    duplicate_and_append(in,out); // duplique in en out

    #ifdef DEBUG_LIST
    print_list_ptrop(out, ea, "trace_backward  in:");
    #endif

    if (flag == NEXT_RET)
      ret_from_call = true ;

    // ------------------------------
    // ajustement des ops normalises
    // ------------------------------

    adjust_function_operands_backward(ea,flag,out,prev_ea,depth);

    // ------------------------------
    // traitement principal
    // ------------------------------

    callui(ui_setstate,st_Work);
    ua_ana0(ea);
    showAddr(ea);


    while (not_basic_block_extremity && !out.empty())
    {

      liste_op event[5];

      // 6 oprandes
      df_op_t *op[6] = {NULL, NULL, NULL, NULL, NULL, NULL};

      bool del_op[6] = {true, true, true, true, true, true};

      bool xchg_1 = false, xchg_2 = false;

      // nombre d'oprandes effectivement utilises
      ushort nb_op = loadOperands(ea, op); // charge les oprandes

      // trouve les lments de traced correspondant aux oprandes
      if (nb_op)
      {
        // parcours de la liste  tracer
        for (liste_ptrop::iterator i_out = out.begin();
             i_out != out.end();
             ++i_out )
        {
          // parcours des oprandes
          for (int i=0; i < nb_op; ++i)
          {
            df_op_t *cop = *i_out;

            // dtection des oprandes de pile  effacer

            // ne pas dtecter si on revient d'une fonction
            if (ret_from_call)
              ret_from_call = false;
            else
              mark_dead_operand(ea+cmd.size, get_func(ea), cop);

            // si l'lment courant de traced contient l'oprande courante

            if ( cop->operator>=(*op[i]) )
            {
              // ne pas indiquer deux fois la ligne
              // si les deux oprandes sont utilises
              if (!counter || result.altval(counter-1) != ea)
                 result.altset(counter++,ea);

              // dtecte certains cas particuliers
              special_ana(i, op, destroyed, xchg_1, xchg_2);

              // passe les oprandes immdiate aucun intert en backward
              if ((*i_out)->type == o_imm)
              {
                post_remove.push_front(cop);
                continue;
              }

              //-----------------------------------------------
              // Op_Is_Destroyed
              //-----------------------------------------------

              if (Op_Is_Destroyed(i+1) || destroyed)
              {
                // ajoute l'vnement
                if ( find(event[3].begin(), event[3].end(), *op[i])
                     == event[3].end() )
                {
                  event[3].push_front(*op[i]);
                }

                // ajoute l'op. cour.  la liste des lments  retirer de out
                potential_remove.push_front(cop);

                if (Op_Is_Changed(i+1) && !destroyed)
                {
                  for (int j = 0; j < nb_op; ++j)
                  {
                    if (i!=j)
                    {
                      if (Op_Is_Aliased(j+1))
                      {
                        // dcouper l'oprande
                        // et mettre les bouts dans post_add

                        if (   (op[j]->type == o_displ)
                            || (op[j]->type == o_phrase) )
                        {
                          liste_ptrop regs;
                          op[j]->createRegsFromPhrase(regs);

                          for (liste_ptrop::iterator t = regs.begin();
                               t != regs.end();
                               ++t)
                          {
                            if(   (find(post_add.begin(), post_add.end(), *t)
                                   == post_add.end())
                               && (find(out.begin(), out.end(), *t)
                                   == out.end()) )
                            {
                              post_add.push_front(*t);
                            }
                            else
                            {
                              post_remove.push_front(*t);
                            }
                          }
                        }

                      }
                      // l'oprande destructrice n'est pas de type o_imm
                      else if(Op_Is_Copied(j+1) && op[j]->type != o_imm)
                      {
                        if(   (find(post_add.begin(), post_add.end(), op[j])
                               == post_add.end())
                           && (find(out.begin(), out.end(), op[j])
                               == out.end()))
                        {
                          // ajoute ce par quoi elle a t dtruite dans out
                          post_add.push_front(op[j]);
                          del_op[j] = false;
                        }
                      }
                    }
                  }
                }

                destroyed = false;
              }

              //-----------------------------------------------
              if (Op_Is_Changed(i+1))
              {
                // ajoute l'vnement
                if ( find(event[1].begin(), event[1].end(), *op[i])
                     == event[1].end() )
                {
                  event[1].push_front(*op[i]);
                }
                // amlioration possible :
                // ajouter dans out l'op qui modifie l'op change
              }

              //-----------------------------------------------
              if (Op_Is_Copied(i+1))
              {
                // ajoute l'vnement
                if ( find(event[2].begin(), event[2].end(), *op[i])
                     == event[2].end() )
                {
                  event[2].push_front(*op[i]);
                }
              }

              //-----------------------------------------------
              if (Op_Is_Aliased(i+1))
              {
                // ajoute l'vnement
                if ( find(event[0].begin(), event[0].end(), *op[i])
                     == event[0].end() )
                {
                  event[0].push_front(*op[i]);
                }
              }

              //-----------------------------------------------
              if (Op_Is_Used(i+1))
              {
                // ajoute l'vnement
                if ( find(event[4].begin(), event[4].end(), *op[i])
                     == event[4].end() )
                {
                  event[4].push_front(*op[i]);
                }
              }
            }
          }

        }

      }

      //-----------------------------------------------
      // Post-traitements
      //-----------------------------------------------

      // ajoute les oprandes marques dans post_remove (type == o_void)
      for(liste_ptrop::iterator p = out.begin(); p != out.end(); ++p)
      {
        if ((*p)->type == o_void)
        {
          post_remove.push_front(*p);
        }
      }

      if (xchg_1 && xchg_2)
        potential_remove.clear(); // si on swap
      else
        post_remove.splice(post_remove.end(), potential_remove);

      // post_remove contient la liste des ops  effacer

      liste_ptrop templ;
      for (liste_ptrop::iterator p = post_remove.begin();
           p!= post_remove.end();
           ++p)
      {
        basic_ana::add_to_dependence_list(*p,templ);
      }
      post_remove.clear();

      // templ contient la liste tendue des ops  effacer


      // retire de out les oprandes dtruites
      delete_list(out, templ);

      // ajoute  out les copie et alias cres par l'instruction courante

      out.splice(out.end(), post_add);

      //-----------------------------------------------
      // commentaires
      //----------------------------------------------

      create_result(event, result, counter);

      //-----------------------------------------------
      // dtruit les oprandes non utilises
      //-----------------------------------------------

      for (int i = 0; i < nb_op; ++i)
      {
     	  if (del_op[i])
     	    delete op[i];
      }

      //-----------------------------------------------
      // incrmentation et dtection de la fin du bloc de base
      //-----------------------------------------------

      // dtection de la fin du bloc de base

      if (is_basic_block_end(ea,true))
        not_basic_block_extremity = false;
      else
      {
        ea = decode_prev_insn(ea); // incrmentation
        showAddr(ea);
      }
    }

    result.altset(-1,counter); // nombre de lignes du rsultat

    ea = reach_end_of_block(ea,true);

    //-----------------------------------------------
    // Liens entre les blocs de base
    // ____________________________
    //
    // rempli et positionne depth, next, flag, prev_ea et callstate
    //-----------------------------------------------

    link_backward(ea, depth, callstate, next, flag, prev_ea);

    #ifdef DEBUG_LIST
    print_list_ptrop(out, ea, "trace_backward out:");
    #endif

  }

  //--------------------------------------------------------------------------
  // duplique une liste_ptrop
  // partie rcursive ne pas appeler directement

  void _duplicate_and_append(const liste_ptrop &in,
                                    liste_ptrop &out,
                                    liste_ptrop &done,
                                    df_op_t *prev_in,
                                    df_op_t *prev_out)
  {

    // parcours de la liste source
    for(liste_ptrop::const_iterator p = in.begin(); p != in.end(); ++p)
    {
      df_op_t *op_in = *p;

      if ( find(done.begin(), done.end(), op_in) == done.end() )
      {                                  // si on a pas dj dupliqu cet lem
        done.push_front(op_in);
        df_op_t *op_out = new df_op_t(*op_in); // duplique l'op courante
        out.push_back(op_out); // et la met dans la destination

        if (prev_in != NULL && prev_out != NULL)
        { // il faut lier les ops entre elles

          if (prev_in->aliased == op_in) // si on vient de (1)
          {
            prev_out->setAliased(op_out);
            op_out->addAlias(prev_out);
          }

          if (    prev_in->aliases != NULL
               && find(prev_in->aliases->begin(),prev_in->aliases->end(),op_in)
                  != prev_in->aliases->end() ) // si on vient de (2)
          {
            prev_out->addAlias(op_out);
            op_out->setAliased(prev_out);
          }
        }

        if (op_in->aliased != NULL)
        {
          liste_ptrop _in ;
          _in.push_front(op_in->aliased);
          _duplicate_and_append(_in, out,done, op_in, op_out); //(1)
        }

        if (op_in->aliases != NULL)
        { //(2)
          _duplicate_and_append(*(op_in->aliases), out,done, op_in, op_out);
        }
      }
    }

  }

  //--------------------------------------------------------------------------
  // duplique une liste_ptrop
  // in : liste  dupliquer
  // out : liste dupliquee

  void duplicate_and_append(const liste_ptrop &in, liste_ptrop &out)
  {
    _duplicate_and_append(in, out, liste_ptrop(), NULL, NULL);
  }
  //--------------------------------------------------------------------------
  // ajoute dans 'liste' toutes les adresses des instruction de retour
  // de fonction de la fonction en ea
  void getRetFromFunc(ea_t ea, list<ea_t>& liste)
  {
    func_t* f = get_func(ea);

    if (f)
    {
      ua_ana0(f->startEA);
      for (ea_t fcode = f->startEA; fcode < f->endEA; )
      {
        if ( Ins_Op_Test(CF_STOP) && (get_first_cref_from(fcode) == BADADDR) )
        {
          liste.push_front(fcode);
        }
        fcode += cmd.size;

        // permet d'viter les data au milieu du code
        // (jump table d'un switch par ex)
        while (!ua_ana0(fcode))
          ua_ana0(++fcode);
      }
    }
  }

  //--------------------------------------------------------------------------
  // Ajoute  liste
  // les diffrentes adresses de retour possibles de la fonction en ea
  // traage avant uniquement

  void getRetAddr(/*IN*/ea_t ea, /*OUT*/list<ea_t> &liste)
  {
    func_t* f = get_func(ea);
    if (f)
    {
      // parcours des ins prcdant le dbut de la fonction
      for (ea_t code_ref = get_first_fcref_to(f->startEA);
           code_ref != BADADDR;
           code_ref = get_next_fcref_to(f->startEA,code_ref))
      {
        if (code_ref != BADADDR)
        {
          ua_ana0(code_ref);
          func_t* f2 = get_func(code_ref);
          if (f2)
          {
            if(code_ref+cmd.size < f2->endEA)
            {
              liste.push_front(code_ref+cmd.size);
            }
          }
        }
      }
    }
  }

} // namespace
