/*
Copyright (c) 2003, Michel Jean-Franois <jfmichel(at)operamail(dot)com>
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer. 

- 	Redistributions in binary form must reproduce the above copyright 
	notice, this list of conditions and the following disclaimer in the 
	documentation and/or other materials provided with the distribution. 

-	Neither the name of Sobek nor the names of its contributors 
	may be used to endorse or promote products derived from this software 
	without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/


#include <ida.hpp>
#include <idp.hpp>
#include <loader.hpp>

#include "options.hpp"
#include "global_ana.hpp"

liste_node* gti; // liste des resultats

unsigned long leak = 0;
unsigned long total = 0;

//--------------------------------------------------------------------------
// This callback is called for UI notification events
static int sample_callback(void * /*user_data*/, int event_id, va_list /*va*/)
{
  if ( event_id != ui_msg )     // avoid recursion
    if ( event_id != ui_setstate
      && event_id != ui_showauto
      && event_id != ui_refreshmarked ) // ignore uninteresting events
                    msg("ui_callback %d\n", event_id);
  return 0;                     // 0 means "process the event"
                                // otherwise the event would be ignored
}

//--------------------------------------------------------------------------
//
//      Initialize.
//
//      IDA will call this function only once.
//      If this function returns PLGUIN_SKIP, IDA will never load it again.
//      If this function returns PLUGIN_OK, IDA will unload the plugin but
//      remember that the plugin agreed to work with the database.
//      The plugin will be loaded again if the user invokes it by
//      pressing the hotkey or selecting it from the menu.
//      After the second load the plugin will stay on memory.
//      If this function returns PLUGIN_KEEP, IDA will keep the plugin
//      in the memory. In this case the initialization function can hook
//      into the processor module and user interface notification points.
//      See the hook_to_notification_point() function.
//
//      In this example we check the input file format and make the decision.
//      You may or may not check any other conditions to decide what you do:
//      whether you agree to work with the database or not.
//
int init(void)
{
//  if ( inf.filetype == f_ELF ) return PLUGIN_SKIP;

// Please uncomment the following line to see how the notification works
//  hook_to_notification_point(HT_UI, sample_callback, NULL);

// Please uncomment the following line to see how the user-defined prefix works
//  set_user_defined_prefix(prefix_width, get_user_defined_prefix);

  if (ph.id != PLFM_386) return PLUGIN_SKIP;
 
  return PLUGIN_KEEP;
}

//--------------------------------------------------------------------------
//      Terminate.
//      Usually this callback is empty.
//      The plugin should unhook from the notification lists if
//      hook_to_notification_point() was used.
//
//      IDA will call this function when the user asks to exit.
//      This function won't be called in the case of emergency exits.

void term(void)
{
  #ifdef DEBUG_LEAK
  if (leak)
    warning("Operands leaked : %d/%d\n",leak,total);
  #endif

  // dtruit le rsultat de l'analyse
  if (gti)
  {
     for(liste_node::iterator i_node = gti->begin(); i_node != gti->end(); )
     {
       netnode n = *(i_node++);
       n.kill();
     }
     gti->clear();
  }
  unhook_from_notification_point(HT_UI, sample_callback);
}


//--------------------------------------------------------------------------
// column widths
static const int widths[] = { 8, 32, 16, 16, 16, 16, 16};

// column headers
static const char *header[] =
{
  "Address",
  "Instruction",
  "Aliased",
  "Changed",
  "Copied",
  "Destroyed",
  "Used",
};

static const char* title = "Sobek Data Flow Analysis v0.1";


//-------------------------------------------------------------------------
// chaque netnode de gti est cod de telle manire :
// altval(-1) = le nombre de lignes du netnode
// altval(x) pour x >-1 : adresse numro x
// supval(x,['0'..'4']) : reprsente les infos sous forme de chane de car.
//                        de l'adresse numro x
// valeur du tag :
// '0' = DF_ALIAS
// '1' = CF_CHG
// '2' = DF_COPY
// '3' = DF_DESTR
// '4' = CF_USE
//-------------------------------------------------------------------------
// function that returns number of lines in the list
static ulong idaapi sizer(void *obj)
{
  liste_node *lnode = (liste_node *)obj;

  int n = 0;

  for (liste_node::iterator il_node = lnode->begin();
       il_node != lnode->end();
       il_node++)
  {
    n += il_node->altval(-1) + 2; // +2 pour chaque ligne entre les basic block
  }
  return (n);
}

//-------------------------------------------------------------------------
// function that generates the list line
static void idaapi desc(void *obj,ulong n,char * const *arrptr)
{
  liste_node *lnode = (liste_node *)obj;
  liste_node::iterator il_node;
  int m = n;

  if ( n == 0 ) // generate the column headers
  {
    for ( int i=0; i < qnumber(header); i++ )
      strcpy(arrptr[i], header[i]);
    return;
  }
  else
    m--;

  for ( il_node = lnode->begin();
       (il_node != lnode->end()) && (m > (il_node->altval(-1)+1));
        il_node++)
    m -= il_node->altval(-1) + 2;

  if (il_node != lnode->end())
  {
    if (m)
    {
      if (m == il_node->altval(-1) + 1) // affiche une sparation
      {
        char buf[MAXSTR];
        a2funcoff(il_node->altval( il_node->altval(-1)-1 ), buf, MAXSTR);
        qsnprintf(arrptr[0], MAXSTR, "%s", buf);
        qsnprintf(arrptr[1], MAXSTR, "________________________________");
      }
      else // affiche la ligne du rsultat
      {
        ea_t ea = il_node->altval(m-1);

        // gnre une chane de caractres reprsentant l'instruction en ea
        
        #ifdef SDK_46
        generate_disasm_line(ea, arrptr[1], MAXSTR);
        #else
        generate_disasm_line(ea, arrptr[1]);
        #endif
        
        tag_remove(arrptr[1], arrptr[1], MAXSTR);  // remove the color coding
        char buf[MAXSTR];
        // transforme une adresse en nom_de_fonction+dplacement
        a2funcoff(ea,buf,MAXSTR);
        qsnprintf(arrptr[0], MAXSTR, "%s", buf); // crit la reprsentation

        for (int i = 0; i < 5; i++) // parcours des attributs
        {
          if (il_node->supval(m-1,char(i+'0')))
            qsnprintf(arrptr[2+i],              // crit la reprsentation
                      MAXSTR, "%s",
                      il_node->supval(m-1,char(i+'0')));
        }
      }
    }
    else
    {
      char buf[MAXSTR];
      a2funcoff(il_node->altval(0),buf,MAXSTR);
      qsnprintf(arrptr[0], MAXSTR, "%s", buf); // affiche une sparation
    }
  }
}

//-------------------------------------------------------------------------
// fonction appele quand l'utilisateur appuie sur enter
// permet de sauter  l'adresse de la ligne slectionne dans le listing
static void idaapi enter_cb(void *obj,ulong n)
{
  liste_node *lnode = (liste_node *)obj;
  liste_node::iterator il_node;
  int m = n-1;

  for ( il_node = lnode->begin();
       (il_node != lnode->end()) && (m > (il_node->altval(-1)+1));
        il_node++)
    m -= il_node->altval(-1) + 2;

  if (il_node != lnode->end())
  {
    if (m)
    {
      if (m == il_node->altval(-1) + 1)
      {
        jumpto(il_node->altval(il_node->altval(-1)-1 ));
      }
      else
      {
        jumpto(il_node->altval(m-1));
      }
    }
    else
    {
      jumpto(il_node->altval(0));
    }
  }
}
//-------------------------------------------------------------------------
// function that is called when the window is closed
static void idaapi destroy_cb(void * /*obj*/)
{

}

//--------------------------------------------------------------------------
// retourne le numro de l'oprande (0 ou 1) slectionne par le curseur
int getOpNum ()
{
  int* x = new int;
  int* y = new int;
  int i;
  // prend la chane de caractres reprsentant l'instruction courante
  char *txt = get_curline();
  char* txt2 = new char[MAXSTR];
  tag_remove(txt,txt2,MAXSTR); // retire les tags de couleurs
  get_cursor(x,y); // prend les coordonnes du curseur

  // cherche la position de la virgule
  for (i = 0; i < MAXSTR && txt2[i] != ','; i++)
    ;
  if (*x >i)
    return (1);
  else
    return (0);
}

//--------------------------------------------------------------------------
//
//      The plugin method
//
//      This is the main function of plugin.
//
//      It will be called when the user selects the plugin.
//
//              arg - the input argument, it can be specified in
//                    plugins.cfg file. The default is zero.
//
//

void run(int arg)
{
  df_op_t *op;

  netnode res;
  res.create();

  int num;

  //----------------------------------------------------------------------
  // analyse l'instruction  la position courante du curseur
  ua_ana0(get_screen_ea());

  if (get_nb_op()) // si l'instruction courante a au moins une oprande
  {
    num = getOpNum();

    if ( !(cmd.Operands[0].showed()) )
       num++;

    op = new df_op_t(num);
    op->normalize();

    msg ("---------------------------------------------------\n"
         "Sobek Data Flow Plugin v0.1\n"
         "Copyright (C) 2003, Michel Jean-Franois\n"
         "---------------------------------------------------\n\n");
      
    basic_ana::set_option(ANA_OPTIONS);
    
    global_ana::set_depth_limit(DEPTH_MIN,DEPTH_MAX);
    global_ana::add_limits(area_t(EA_MIN,EA_MAX));


    msg("Please wait...\n");

    if (arg)
    {
      gti = global_ana::global_trace(get_screen_ea(),op,TRACE_BACKWARD);
    }
    else
    {
      gti = global_ana::global_trace(get_screen_ea(),op,TRACE_FORWARD);
    }

    //--------------------------------------------------------------------
    // Affiche le rsultat

    msg ("Building list...\n");

    choose2(false,        // non-modal window
    -1, -1, -1, -1,       // position is determined by Windows
    gti,                  // pass the created netnode to the window
    qnumber(header),      // number of columns
    widths,               // widths of columns
    sizer,                // function that returns number of lines
    desc,                 // function that generates a line
    title,                // window title
    -1,                   // use the default icon for the window
    0,                    // position the cursor on the first line
    NULL,                 // "kill" callback
    NULL,                 // "new" callback
    NULL,                 // "update" callback
    NULL,                 // "edit" callback
    enter_cb,             // function to call
                          // when the user pressed Enter
    NULL,                 // function to call
                          // when the window is closed
    NULL,                 // use default popup menu items
    NULL);                // use the same icon for all lines
  }
  // else
  //   no op
}

//--------------------------------------------------------------------------
char comment[] = "Sobek Data Flow Analysis plugin v0.1";

char help[] =
	"Sobek is a simple data flow analysis plugin\n"
	"It helps the reverse engineering process by answering two questions :\n"
	"\n"
	"Trace Forward : show following instructions where the selected operand is used and/or propagated.\n"
	"Trace Backward : show preceding instructions from which depends the value of the selected operand.\n"; 

//--------------------------------------------------------------------------
// This is the preferred name of the plugin module in the menu system
// The preferred name may be overriden in plugins.cfg file

char wanted_name[] = "Sobek Forward";


// This is the preferred hotkey for the plugin module
// The preferred hotkey may be overriden in plugins.cfg file
// Note: IDA won't tell you if the hotkey is not correct
//       It will just disable the hotkey.

char wanted_hotkey[] = "Shift-1";


//--------------------------------------------------------------------------
//
//      PLUGIN DESCRIPTION BLOCK
//
//--------------------------------------------------------------------------

extern "C" plugin_t PLUGIN = {
  IDP_INTERFACE_VERSION,
  PLUGIN_KEEP,           // plugin flags
  init,                 // initialize

  term,                 // terminate. this pointer may be NULL.

  run,                  // invoke plugin

  comment,              // long comment about the plugin
                        // it could appear in the status line
                        // or as a hint

  help,                 // multiline help about the plugin

  wanted_name,          // the preferred short name of the plugin
  wanted_hotkey         // the preferred hotkey to run the plugin
};
